
class ToplevelInstaller < Installer

  TASKS = [
    [ 'config',   'saves your configurations' ],
    [ 'show',     'shows current configuration' ],
    [ 'setup',    'compiles extention or else' ],
    [ 'install',  'installs files' ],
    [ 'clean',    "does `make clean' for each extention" ]
  ]


  def initialize( argv, root )
    @argv = argv
    super nil, {'verbose' => true}, root, '.'

    Installer.declear_toplevel_installer self
  end


  def execute
    argv = @argv
    @argv = nil
    task = parsearg_global( argv )

    case task
    when 'config'
      @config = ConfigTable.create
    else
      @config = ConfigTable.load
    end
    parsearg_TASK task, argv

    exectask task
  end

  def exectask( task )
    if task == 'show' then
      exec_show
    else
      try task
    end
  end

  def try( task )
    $stderr.printf "#{File.basename $0}: entering %s phase...\n", task if verbose?
    begin
      __send__ 'exec_' + task
    rescue
      $stderr.printf "%s failed\n", task
      raise
    end
    $stderr.printf "#{File.basename $0}: %s done.\n", task if verbose?
  end

  #
  # processing arguments
  #

  def parsearg_global( argv )
    task_re = /\A(?:#{TASKS.collect {|i| i[0] }.join '|'})\z/

    while arg = argv.shift do
      case arg
      when /\A\w+\z/
        task_re === arg or raise InstallError, "wrong task: #{arg}"
        return arg

      when '-q', '--quiet'
        @options['verbose'] = false

      when       '--verbose'
        @options['verbose'] = true

      when '-h', '--help'
        print_usage $stdout
        exit 0

      when '-v', '--version'
        puts "#{File.basename $0} version #{Version}"
        exit 0
      
      when '--copyright'
        puts Copyright
        exit 0

      when nil
        raise InstallError, 'no task given'

      else
        raise InstallError, "unknown global option '#{arg}'"
      end
    end
  end


  def parsearg_TASK( task, argv )
    mid = "parsearg_#{task}"
    if respond_to? mid, true then
      __send__ mid, argv
    else
      argv.empty? or
          raise InstallError, "#{task}:  unknown options: #{argv.join ' '}"
    end
  end

  def parsearg_config( args )
    re = /\A--(#{ConfigTable.keys.join '|'})=/
    @options['config-opt'] = []

    while i = args.shift do
      if /\A--?\z/ === i then
        @options['config-opt'] = args
        break
      end
      m = re.match(i) or raise InstallError, "config: unknown option #{i}"
      @config[ m[1] ] = m.post_match
    end
  end

  def parsearg_install( args )
    @options['no-harm'] = false
    args.each do |i|
      if i == '--no-harm' then
        @options['no-harm'] = true
      else
        raise InstallError, "install: unknown option #{i}"
      end
    end
  end


  def print_usage( out )
    out.puts
    out.puts 'Usage:'
    out.puts "  ruby #{File.basename $0} <global option>"
    out.puts "  ruby #{File.basename $0} <task> [<task options>]"

    fmt = "  %-20s %s\n"
    out.puts
    out.puts 'Global options:'
    out.printf fmt, '-q,--quiet',   'suppress message outputs'
    out.printf fmt, '   --verbose', 'output messages verbosely'
    out.printf fmt, '-h,--help',    'print this message'
    out.printf fmt, '-v,--version', 'print version and quit'
    out.printf fmt, '--copyright',  'print copyright and quit'

    out.puts
    out.puts 'Tasks:'
    TASKS.each do |name, desc|
      out.printf "  %-10s  %s\n", name, desc
    end

    out.puts
    out.puts 'Options for config:'
    ConfigTable::DESCRIPTER.each do |name, (default, arg, desc, default2)|
      default = default2 || default
      out.printf "  %-20s %s [%s]\n", "--#{name}=#{arg}", desc, default
    end
    out.printf "  %-20s %s [%s]\n",
        '--rbconfig=path', 'your rbconfig.rb to load', "running ruby's"

    out.puts
    out.puts 'Options for install:'
    out.printf "  %-20s %s [%s]\n",
        '--no-harm', 'only display what to do if given', 'off'

    out.puts
  end

  #
  # config
  #

  def exec_config
    super
    @config.save
  end

  #
  # show
  #

  def exec_show
    ConfigTable.each_name do |k|
      v = @config.noproc_get(k)
      if not v or v.empty? then
        v = '(not specified)'
      end
      printf "%-10s %s\n", k, v
    end
  end

end

