/**
 * @file   qnengo.cpp
 * @author NOKUBI Takatsugu <knok@daionet.gr.jp>
 * @date   Thu May  1 12:17:07 2003
 * 
 * @brief  QNengo main widget
 * 
 * $Id: qnengo.cpp,v 1.2 2003/05/03 02:10:30 knok Exp $
 *
 * Copyright (C) 2003 NOKUBI Takatsugu All rights reserved.
 * This is free software with ABSOLUTELY NO WARRANTY.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA
 *
 */

#include "qnengo.h"

#include <qframe.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qaction.h>
#include <qcombobox.h>
#include <qclipboard.h>
#include <qabstractlayout.h>
#include <qpushbutton.h>
#include <qmessagebox.h>

#include <qpeapplication.h>
#include <qpetoolbar.h>
#include <qpemenubar.h>

QNengo::QNengo(QWidget *parent, const char *name, int WFlags)
    : QMainWindow(parent, name, WFlags)
{
    setName("QNengo");
    setCaption(tr("QNengo"));
    setToolBarsMovable(FALSE);
    this->setFont(QFont("unifont"));

    QPEToolBar *toolBar = new QPEToolBar(this, "tool");
    toolBar->setHorizontalStretchable(FALSE);
    addToolBar(toolBar, "tool", QMainWindow::Top, TRUE);
    QPEMenuBar *menuBar = new QPEMenuBar(toolBar);
    QPopupMenu *listMenuFile = new QPopupMenu(menuBar);
    QPopupMenu *listAbout = new QPopupMenu(menuBar);

    menuBar->insertItem(tr("File"), listMenuFile);
    actExit = new QAction(tr("Exit"), QString::null, 0, this, 0);
    connect(actExit, SIGNAL(activated()), this, SLOT(exitapp()));
    actExit->addTo(listMenuFile);

    QPopupMenu *listMenuCopy = new QPopupMenu(menuBar);
    menuBar->insertItem(tr("Copy"), listMenuCopy);
    actCopyAD = new QAction(tr("A.D string"), QString::null, 0, this, 0);
    connect(actCopyAD, SIGNAL(activated()), this, SLOT(copyAD()));
    actCopyAD->addTo(listMenuCopy);
    actCopyNengo = new QAction(tr("Nengo string"), QString::null, 0, this, 0);
    connect(actCopyNengo, SIGNAL(activated()), this, SLOT(copyNengo()));
    actCopyNengo->addTo(listMenuCopy);

    menuBar->insertItem(tr("About"), listAbout);
    actAbout = new QAction("About", QString::null, 0, this, 0);
    connect(actAbout, SIGNAL(activated()), this, SLOT(about()));
    actAbout->addTo(listAbout);

    /// main frame
    topF = new QFrame(this);
    setCentralWidget(topF);
    topL = new QVBoxLayout(topF);
    topL->setSpacing(0);
    topL->setMargin(10);

    /// upper frame
    upperF = new QGroupBox(topF, "upper");
    upperF->setTitle(tr("A.D."));
    upperL = new QHBoxLayout(upperF);
    QSpacerItem *tmpspc = new QSpacerItem(40, 40);
    upperL->addItem(tmpspc);
    century = new QComboBox(upperF, "century");
    century->insertItem("19");
    century->insertItem("20");
    connect(century, SIGNAL(activated(int)), SLOT(updateAD()));
    upperL->addWidget(century);
    yten = new QComboBox(upperF, "ten years");
    for (int i = 0; i <= 9; i ++) {
	QString s("%1");
	yten->insertItem(s.arg(i));
    }
    connect(yten, SIGNAL(activated(int)), SLOT(updateAD()));
    upperL->addWidget(yten);
    year = new QComboBox(upperF, "years");
    for (int i = 0; i <= 9; i ++) {
	QString s("%1");
	year->insertItem(s.arg(i));
    }
    connect(year, SIGNAL(activated(int)), SLOT(updateAD()));
    upperL->addWidget(year);
    tmpspc = new QSpacerItem(40, 40);
    upperL->addItem(tmpspc);
    topL->addWidget(upperF);

    /// downer frame
    downerF = new QGroupBox(topF, "downer");
    downerF->setTitle(tr("Wareki"));
    downerL = new QHBoxLayout(downerF);
    topL->addWidget(downerF);
    tmpspc = new QSpacerItem(20, 20);
    downerL->addItem(tmpspc);
    nengo = new QComboBox(downerF, "nengo");
    nengo->insertItem(tr("Meiji"));
    nengo->insertItem(tr("Taishou"));
    nengo->insertItem(tr("Syouwa"));
    nengo->insertItem(tr("Heisei"));
    connect(nengo, SIGNAL(activated(int)), SLOT(updateNengo()));
    downerL->addWidget(nengo);
    nyten = new QComboBox(downerF, "ten years");
    for (int i = 0; i <= 9; i ++) {
	QString s("%1");
	nyten->insertItem(s.arg(i));
    }
    connect(nyten, SIGNAL(activated(int)), SLOT(updateNengo()));
    downerL->addWidget(nyten);
    nyear = new QComboBox(downerF, "years");
    for (int i = 0; i <= 9; i ++) {
	QString s("%1");
	nyear->insertItem(s.arg(i));
    }
    connect(nyear, SIGNAL(activated(int)), SLOT(updateNengo()));
    downerL->addWidget(nyear);
    tmpspc = new QSpacerItem(20, 20);
    downerL->addItem(tmpspc);

    /// button frame
    btnF = new QFrame(topF, "Button");
    btnL = new QHBoxLayout(btnF);
    topL->addWidget(btnF);
    btnCopyAD = new QPushButton(btnF, "A.D.");
    btnCopyAD->setText(tr("A.D."));
    connect(btnCopyAD, SIGNAL(clicked()), SLOT(copyAD()));
    btnL->addWidget(btnCopyAD);
    btnCopyNengo = new QPushButton(btnF, "Nengo");
    btnCopyNengo->setText(tr("Nengo"));
    connect(btnCopyNengo, SIGNAL(clicked()), SLOT(copyNengo()));
    btnL->addWidget(btnCopyNengo);
    btnQuit = new QPushButton(btnF, "Quit");
    btnQuit->setText(tr("Quit"));
    connect(btnQuit, SIGNAL(clicked()), SLOT(close()));
    btnL->addWidget(btnQuit);

    /// Initialize - 2003 A.D. / Heisei 15
    adyear = 2003;
    ntype = Heisei;
    neyear = 15;
    setWidgetAD();
    setWidgetNengo();
}

void QNengo::setWidgetAD()
{
    if (adyear < 1900 || adyear >= 2100)
	return;
    int c = adyear - 1900;
    c /= 100;
    century->setCurrentItem(c);
    int t = adyear - (c + 19) * 100;
    t /= 10;
    yten->setCurrentItem(t);
    int y = adyear - (c + 19) * 100 - t * 10;
    year->setCurrentItem(y);
}

void QNengo::setWidgetNengo()
{
    nengo->setCurrentItem((int) ntype);
    int ty = neyear / 10;
    nyten->setCurrentItem(ty);
    int y = neyear - ty * 10;
    nyear->setCurrentItem(y);
}

void QNengo::ADtoNengo()
{
    /// Heisei: since 1989-
    if (adyear >= 1989) {
	ntype = Heisei;
	neyear = adyear - 1989 + 1;
	return;
    }
    /// Syouwa: since 1926-
    if (adyear >= 1926) {
	ntype = Syouwa;
	neyear = adyear - 1926 + 1;
	return;
    }
    /// Taishou: since 1912-
    if (adyear >= 1912) {
	ntype = Taishou;
	neyear = adyear - 1912 + 1;
	return;
    }
    /// Meiji: since 1868-
    ntype = Meiji;
    neyear = adyear - 1868 + 1;
}

void QNengo::fixNengo()
{
    if (neyear < 1) {
	neyear = 1;
	return;
    }
    int max = 100;
    switch (ntype) {
    case Syouwa:
	max = 63;
	break;
    case Taishou:
	max = 14;
	break;
    case Meiji:
	max = 44;
	break;
    default:
	break;
    }
    if (neyear > max)
	neyear = max;
}

void QNengo::NengoToAD()
{
    fixNengo();
    switch (ntype) {
    case Heisei:
	adyear = 1989 + neyear - 1;
	break;
    case Syouwa:
	adyear = 1926 + neyear - 1;
	break;
    case Taishou:
	adyear = 1912 + neyear - 1;
	break;
    case Meiji:
	adyear = 1868 + neyear - 1;
    }
}

void QNengo::updateFromAD()
{
    /// adyear ʬ
    int c = adyear / 100 - 19;
    int t = (adyear - (c + 19) * 100) / 10;
    int y = adyear - (c + 19) * 100 - t * 10;
    /// widget 
    century->setCurrentItem(c);
    yten->setCurrentItem(t);
    year->setCurrentItem(y);
}

void QNengo::updateFromNengo()
{
    /// neyear ʬ
    int t = neyear / 10;
    int y = neyear - t * 10;
    /// widget 
    nengo->setCurrentItem((int) ntype);
    nyten->setCurrentItem(t);
    nyear->setCurrentItem(y);
}

void QNengo::updateAD()
{
    /// AD widget -> adyear
    int c = century->currentItem();
    int t = yten->currentItem();
    int y = year->currentItem();
    adyear = (c + 19) * 100 + t * 10 + y;
    /// adyear -> nengo
    ADtoNengo();
    /// nengo -> nengo widget
    updateFromNengo();
}

void QNengo::updateNengo()
{
    /// nengo widget -> nengo
    int n = nengo->currentItem();
    int t = nyten->currentItem();
    int y = nyear->currentItem();
    ntype = (Nengo) n;
    neyear = t * 10 + y;
    /// fix nengo
    fixNengo();
    /// nengo -> nengo widget
    updateFromNengo();
    /// nengo -> adyear
    NengoToAD();
    /// adyear -> AD widget
    updateFromAD();
}

void QNengo::copyAD()
{
    QClipboard *cp = QPEApplication::clipboard();
    QString s("%1");
    cp->setText(s.arg(adyear));
}

void QNengo::copyNengo()
{
    QClipboard *cp = QPEApplication::clipboard();
    QString s("%1.%2"), n;
    switch (ntype) {
    case Heisei:
	n = "H";
	break;
    case Syouwa:
	n = "S";
	break;
    case Taishou:
	n = "T";
	break;
    case Meiji:
	n = "M";
	break;
    }
    cp->setText(s.arg(n).arg(neyear));
}

void QNengo::exitapp()
{
    close();
}

void QNengo::about()
{
  QMessageBox::information(this, "QNengo",
                           "QNengo Version. " VERSION "\n"
                           "Copyright (c) 2003\nNOKUBI Takatsugu\n\n"
                           "Qtrains comes with\nABSOLUTELY NO \n WARRANTY.\n"
                           "This software is \nreleased under GNU GPL.\n"
                           );
}
