// IMKit-Anthy: A Qtopia InputMethod interface for Anthy
// Copyright (C) 2002  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_ANTHY_0_3_2 $
// $Id: util.h,v 1.8 2002/07/12 11:51:46 yamaken Exp $

#ifndef IMKIT_UTIL_H
#define IMKIT_UTIL_H

#include <map>


class Activatable {
protected:
  //Activatable(void);  //饹Ǥset_activity()
  virtual ~Activatable(void);
  virtual void on_activate(void);
  virtual void on_deactivate(void);
  virtual void set_activity(bool activity) = 0;

public:
  virtual bool is_active(void) const = 0;
  virtual void activate(bool activity = true);
  void deactivate(void);
};

#define IMKIT_DECLARE_SIGNALS_FOR_ACTIVATABLE \
  void slot_activated(bool activity);

#define IMKIT_DECLARE_ON_ACTIVATE_HANDLERS \
  virtual void on_activate(void); \
  virtual void on_deactivate(void);

#define IMKIT_DEFINE_ON_ACTIVATE_HANDLERS(_class) \
  void _class::on_activate(void) { emit slot_activated(true); } \
  void _class::on_deactivate(void) { emit slot_activated(false); }

#define IMKIT_DECLARE_SLOTS_FOR_ACTIVATABLE \
  virtual void slot_activate(bool activity);

#define IMKIT_DEFINE_SLOTS_FOR_ACTIVATABLE(_class) \
  void _class::slot_activate(bool activity) { activate(activity); }


class StdActivatable : public virtual Activatable {
protected:
  bool _is_active;

protected:
  inline StdActivatable(void) : _is_active(false) {}
  virtual ~StdActivatable(void);
  virtual void set_activity(bool activity);

public:
  virtual bool is_active(void) const;
};


template <class KeyType, class ValueType>
class MapWithConstLookup : public map<KeyType, ValueType> {
protected:
  data_type _default_value;

public:
  inline MapWithConstLookup(void) {}
  MapWithConstLookup(data_type default_value_init);
  const data_type &lookup(const key_type &key) const;
  const data_type &default_value(void) const;
  void set_default_value(data_type &new_value);
};


template <class KeyType, class ValueType>
class BidirMap {
public:
  typedef MapWithConstLookup<KeyType, ValueType> ordinary_map_t;
  typedef MapWithConstLookup<ValueType, KeyType> inverse_map_t;
  struct pair_t {
    KeyType key;
    ValueType value;
  };
  static const size_t pair_size = sizeof(pair_t);

public:
  ordinary_map_t ordinary_map;
  inverse_map_t inverse_map;
  
public:
  BidirMap(void);
  BidirMap(pair_t *defs, int n_defs);
  BidirMap(pair_t *defs, int n_defs,
           ordinary_map_t::data_type ordinary_map_default,
           inverse_map_t::data_type inverse_map_default);
  virtual ~BidirMap(void);
  virtual void init_map(pair_t *defs, int n_defs);
  virtual void add_map(KeyType key, ValueType value);
};

//////////////////////////////////////////////////////////////////////////////

template <class KeyType, class ValueType>
MapWithConstLookup<KeyType, ValueType>::
MapWithConstLookup(data_type default_value_init)
  : _default_value(default_value_init)
{
}

template <class KeyType, class ValueType>
const MapWithConstLookup<KeyType, ValueType>::data_type &
MapWithConstLookup<KeyType, ValueType>::lookup(const key_type &key) const {
  if (find(key) != end()) {
    return find(key)->second;
  } else {
    return default_value();
  }
}

template <class KeyType, class ValueType>
const MapWithConstLookup<KeyType, ValueType>::data_type &
MapWithConstLookup<KeyType, ValueType>::default_value(void) const {
  return _default_value;
}

template <class KeyType, class ValueType>
void
MapWithConstLookup<KeyType, ValueType>::set_default_value(data_type &new_value)
{
  _default_value = new_value;
}

template <class KeyType, class ValueType>
BidirMap<KeyType, ValueType>::BidirMap(void) {
}

template <class KeyType, class ValueType>
BidirMap<KeyType, ValueType>::BidirMap(pair_t *defs, int n_defs) {
  init_map(defs, n_defs);
}

template <class KeyType, class ValueType>
BidirMap<KeyType, ValueType>::
BidirMap(pair_t *defs, int n_defs,
         ordinary_map_t::data_type ordinary_map_default,
         inverse_map_t::data_type inverse_map_default)
{
  init_map(defs, n_defs);
  ordinary_map.set_default_value(ordinary_map_default);
  inverse_map.set_default_value(inverse_map_default);
}

template <class KeyType, class ValueType>
BidirMap<KeyType, ValueType>:: ~BidirMap(void) {
}

template <class KeyType, class ValueType>
void
BidirMap<KeyType, ValueType>::init_map(pair_t *defs, int n_defs) {
  int i;

  for (i = 0; i < n_defs; i++) {
    add_map(defs->key, defs->value);
    defs++;
  }
}

template <class KeyType, class ValueType>
void
BidirMap<KeyType, ValueType>::add_map(KeyType key, ValueType value) {
  ordinary_map[key] = value;
  inverse_map[value] = key;
}

#endif  //IMKIT_UTIL_H
