// IMKit-UIM: A Qtopia InputMethod interface for UIM
// Copyright (C) 2002,2003  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_0_PRE6 $
// $Id: engine_uim.cpp,v 1.14 2003/12/27 20:06:48 yamaken Exp $

#include <ctype.h>
#include <string.h>
#include <qtextcodec.h>
#include "uim_keyevent.h"
#include "engine_uim.h"

#if 1
#define IMKIT_UIM_ENCODING "UTF-8"
#else
#define IMKIT_UIM_ENCODING "EUC-JP"
#endif

#ifdef UIM_INPUT_MAP_INVALID
#error "UIM_INPUT_MAP_INVALID is predefined"
#else
#define UIM_INPUT_MAP_INVALID -0x10000
#endif

#if 0
static BidirMap<int, PreeditState>::pair_t n2a_preedit_state_def[] = {
  {UIM_PUSHBACK_ATTR_CONV, IMKIT_PREEDIT_ST_CONV},
  {UIM_PUSHBACK_ATTR_CSEG, IMKIT_PREEDIT_ST_CSEG},
  {UIM_PUSHBACK_ATTR_EDIT, IMKIT_PREEDIT_ST_EDIT},
  {UIM_PUSHBACK_ATTR_NONE, IMKIT_PREEDIT_ST_NONE}
};

BidirMap<int, PreeditState>
UIMEngine::n2a_preedit_state(n2a_preedit_state_def,
                               (sizeof(n2a_preedit_state_def)
                                / BidirMap<int, PreeditState>::pair_size),
                               UIM_PUSHBACK_ATTR_NONE, IMKIT_PREEDIT_ST_NONE);
#endif


UIMSegment::UIMSegment(int attr_init, const char *raw_str = "")
  : attr(attr_init)
{
  sync(raw_str);
}

UIMSegment::~UIMSegment(void) {
}

const QString &
UIMSegment::str(void) {
  return unicode_str;
}

const char *
UIMSegment::raw_str(void) const {
  const char *_raw_str;

  _raw_str = UIMEngine::codec()->fromUnicode(unicode_str);

  return _raw_str;
}

UIMSegment::Type
UIMSegment::type(void) const {
  if (attr & UPeAttr_Cursor) {
    if (attr & UPeAttr_Reverse) {
      return IMKIT_SEG_FOCUS;
    } else {
      return IMKIT_SEG_CURSOR;
    }
  } else if (attr & UPeAttr_UnderLine) {
    return IMKIT_SEG_EDITING;
  } else {
    return IMKIT_SEG_NORMAL;
  }
}

void
UIMSegment::sync(void) {
}

void
UIMSegment::sync(const char *raw_str) {
  if (!raw_str) raw_str = "";
  unicode_str = UIMEngine::codec()->toUnicode(raw_str);
}

UIMCandidate::UIMCandidate(char *uim_str) {
  sync(uim_str);
}

UIMCandidate::~UIMCandidate(void) {
}

const QString &
UIMCandidate::str(void) {
  return unicode_str;
}

void
UIMCandidate::sync(char *uim_str) {
  if (!uim_str) uim_str = "";
  unicode_str = UIMEngine::codec()->toUnicode(uim_str);
}

const QTextCodec *
UIMEngine::codec(void) {
  QTextCodec *_codec;

  _codec = QTextCodec::codecForName(IMKIT_UIM_ENCODING);
  _codec = (_codec) ? _codec : QTextCodec::codecForName("ISO8859-1");

  return _codec;
}

UIMEngine::UIMEngine(void)
  : uim_ctx(NULL), segments(NULL), candidates(NULL),
    n2a_inputmap(NULL, 0, UIM_INPUT_MAP_INVALID, IMKIT_INPUT_MAP_TERM),
    _command_map(NULL)
{
  int err;

  err = uim_init();
  init_state();
  sync();
}

UIMEngine::~UIMEngine(void) {
  free_state();
  imkit_delete_command_map(_command_map);
}

void
UIMEngine::reset(void) {
  //free_state();
  //init_state();
  //uim_reset_context(uim_ctx);
  init_segments();
  init_candidates();
  sync();
}

void
UIMEngine::init_state(void) {
  select_im(IMKIT_INPUT_MAP_RUNTIME_VAL_BASE);
}

void
UIMEngine::free_state(void) {
  if (uim_ctx) {
    uim_release_context(uim_ctx);
    uim_ctx = NULL;
  }
  imkit_delete_ref_container<Segments>(&segments);
  imkit_delete_ref_container<Candidates>(&candidates);
}

void
UIMEngine::init_segments(void) {
  imkit_init_ref_container<Segments>(&segments);
}

void
UIMEngine::init_candidates(void) {
  imkit_init_ref_container<Candidates>(&candidates);
}

void
UIMEngine::update_inputmap(void) {
  int i, n_states;

  n2a_inputmap.init_map(NULL, 0);
  emit map_states_cleared();
  n_states = uim_get_nr_modes(uim_ctx);
  for (i = 0; i < n_states; i++) {
    InputMap abstract_state;
    QString label;

    label = codec()->toUnicode(uim_get_mode_name(uim_ctx, i));

    abstract_state = (InputMap)((int)IMKIT_INPUT_MAP_RUNTIME_VAL_BASE + i);
    if (label == UIMEngine::tr("RAW")
        || label == UIMEngine::tr("chokusetsu-nyuuryoku"))
    {
      abstract_state = IMKIT_INPUT_MAP_ALPHA;
    } else if (label == UIMEngine::tr("hiragana")
               || label == UIMEngine::tr("TUT-hi"))
    {
      abstract_state = IMKIT_INPUT_MAP_HIRAGANA;
    } else if (label == UIMEngine::tr("katakana")
               || label == UIMEngine::tr("TUT-ka"))
    {
      abstract_state = IMKIT_INPUT_MAP_KATAKANA;
    } else if (label == UIMEngine::tr("zenkaku-eisuu")) {
      abstract_state = IMKIT_INPUT_MAP_WALPHA;
    } else if (label == UIMEngine::tr("tcode")
               || label == UIMEngine::tr("py"))
    {
      abstract_state = IMKIT_INPUT_MAP_KANJI;
    }

    n2a_inputmap.add_map(i, abstract_state);
    emit map_state_added(abstract_state, label);
  }
  emit map_changed(map_state());
}

const char *
UIMEngine::name(void) const {
  return "uim";
}

InputMap
UIMEngine::map_state(void) const {
  int native_map_state;

  native_map_state = uim_get_current_mode(uim_ctx);

  return n2a_inputmap.ordinary_map.lookup(native_map_state);
}

CommandMap *
UIMEngine::command_map(void) {
  typedef TemplateCommandWithKeyEvent<UIMEngine> command_t;
  command_t *cmd;

  if (!_command_map) {
    cmd = new command_t(this,
                        &UIMEngine::cmd_receive_keyevent,
                        "receive_keyevent");

    _command_map = new CommandMap;
    (*_command_map)["receive_keyevent"] = cmd;
  }

  return _command_map;
}

void
UIMEngine::update_im_selector(void) {
  int i, n_im;

  emit im_selector_cleared();
  n_im  = uim_get_nr_im(uim_ctx);
  for (i = 0; i < n_im; i++) {
    QString label;

    label = codec()->toUnicode(uim_get_im_name(uim_ctx, i));
    emit im_added((InputMap)i, label);
  }
}

void
UIMEngine::cmd_receive_keyevent(QKeyEvent &e) {
  UIMKeyEvent uim_e(e);
  bool ignored = true;

  if (uim_e.keycode) {
    if (uim_e.is_press) {
      ignored = uim_press_key(uim_ctx, uim_e.keycode, uim_e.modifiers);
    } else {
      ignored = uim_release_key(uim_ctx, uim_e.keycode, uim_e.modifiers);
    }
  }
  if (!ignored) {
    e.accept();
  }
}

void
UIMEngine::input_str(const QString &str) {
  //TODO
}

void
UIMEngine::input_char(QChar chr) {
  //TODO
}

void
UIMEngine::select_im(InputMap nth) {
  char *lang, *engine, *encoding;

  if (uim_ctx && (int)nth < uim_get_nr_im(uim_ctx)) {
    lang = uim_get_im_language(uim_ctx, (int)nth);
    engine = uim_get_im_name(uim_ctx, (int)nth);
  } else {
    lang = engine = NULL;
  }
  if (uim_ctx) {
    uim_release_context(uim_ctx);
  }
  encoding = IMKIT_UIM_ENCODING;

  uim_ctx = uim_create_context(this, encoding, lang, engine, cb_commit);
  uim_set_preedit_cb(uim_ctx, cb_clear, cb_pushback, cb_update);
  uim_set_mode_list_update_cb(uim_ctx, cb_set_mode_list_update);
  uim_set_mode_cb(uim_ctx, cb_set_mode);
  uim_set_candidate_cb(uim_ctx,
                       cb_candidate_win_activate,
                       cb_candidate_win_update_focus,
                       cb_candidate_win_deactivate);
  init_segments();
  init_candidates();

  update_inputmap();
}

void
UIMEngine::select_map(InputMap new_map) {
  int native;

  native = n2a_inputmap.inverse_map.lookup(new_map);
  if (native == UIM_INPUT_MAP_INVALID) return;

  uim_set_mode(uim_ctx, native);
  emit map_changed(new_map);
}

void
UIMEngine::select_candidate(Candidates::size_type nth) {
  uim_set_candidate_index(uim_ctx, nth);
}

void
UIMEngine::cb_commit(void *_this, char *str) {
  ((UIMEngine *)_this)->slot_commit(str);
}

void
UIMEngine::cb_clear(void *_this) {
  ((UIMEngine *)_this)->slot_clear();
}

void
UIMEngine::cb_pushback(void *_this, int attr, char *str) {
  ((UIMEngine *)_this)->slot_pushback(attr, str);
}

void
UIMEngine::cb_update(void *_this) {
  ((UIMEngine *)_this)->slot_update();
}

void
UIMEngine::cb_set_mode(void *_this, int mode) {
  ((UIMEngine *)_this)->slot_set_mode(mode);
}

void
UIMEngine::cb_set_mode_list_update(void *_this) {
  ((UIMEngine *)_this)->slot_set_mode_list_update();
}

void
UIMEngine::cb_candidate_win_activate(void *_this, int nr, int initial_index) {
  ((UIMEngine *)_this)->slot_candidate_win_activate(nr, initial_index);
}

void
UIMEngine::cb_candidate_win_update_focus(void *_this, int index) {
  ((UIMEngine *)_this)->slot_candidate_win_update_focus(index);
}

void
UIMEngine::cb_candidate_win_deactivate(void *_this) {
  ((UIMEngine *)_this)->slot_candidate_win_deactivate();
}

void
UIMEngine::slot_commit(char *str) {
  QString unicode_str;

  unicode_str = codec()->toUnicode(str);
  emit committed(unicode_str);
}

void
UIMEngine::slot_clear(void) {
  init_segments();
}

void
UIMEngine::slot_pushback(int attr, char *str) {
  segments->push_back(new UIMSegment(attr, str));
}

void
UIMEngine::slot_update(void) {
  sync();
}

void
UIMEngine::slot_set_mode(int mode) {
  InputMap abstract_mode;

  abstract_mode = n2a_inputmap.ordinary_map.lookup(mode);
  emit map_changed(abstract_mode);
}

void
UIMEngine::slot_set_mode_list_update(void) {
  update_inputmap();
}

void
UIMEngine::slot_candidate_win_activate(int nr, int initial_index) {
  int i;
  char *str;

  init_candidates();
  for (i = 0; i < nr; i++) {
    str = uim_get_candidate(uim_ctx, i);
    candidates->push_back(new UIMCandidate(str));
    free(str);
  }

  emit update_candidates(*candidates);
  slot_candidate_win_update_focus(initial_index);
  emit activation_hint_for_candidates(true);
}

void
UIMEngine::slot_candidate_win_update_focus(int index) {
  emit hilight_candidate(index, false);
}

void
UIMEngine::slot_candidate_win_deactivate(void) {
  emit activation_hint_for_candidates(false);
}

PreeditState
UIMEngine::preedit_state(void) const {
  Segments::const_iterator segment;

  for (segment = segments->begin(); segment != segments->end(); segment++) {
    if ((*segment)->type() == Segment::IMKIT_SEG_FOCUS) {
      return IMKIT_PREEDIT_ST_CONV;
    }
  }
  return IMKIT_PREEDIT_ST_EDIT;
}

void
UIMEngine::sync(void) {
  emit preedit_state_changed(preedit_state());
  emit segments_changed(*segments);
}
