// IMKit-uim: A Qtopia InputMethod interface for uim
// Copyright (C) 2002-2004  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_0 $
// $Id: uim_keyevent.cpp,v 1.13 2004/03/06 15:27:01 yamaken Exp $

#include <ctype.h>
#include "keyfilter.h"
#include "uim_keyevent.h"

// keysym translation table for uim
//
// See following documents for Zaurus specific keys
//
// SL-A300
// http://developer.ezaurus.com/sl_j/doc/software/keycode_qt_a300_20021213.pdf
//
// SL-B500 and SL-C700
// http://developer.ezaurus.com/sl_j/doc/software/keycode_b500c700.pdf

static BidirMap<Qt::Key, UKey>::pair_t q2u_keycode_def[] = {
  {Qt::Key_Shift,     UKey_Shift_key},
  {Qt::Key_Control,   UKey_Control_key},
  {Qt::Key_Alt,       UKey_Alt_key},
  {Qt::Key_Meta,      UKey_Meta_key},

  {Qt::Key_Escape,    UKey_Escape},
  {Qt::Key_Tab,       UKey_Tab},
  {Qt::Key_Backspace, UKey_Backspace},
  {Qt::Key_Delete,    UKey_Delete},
  {Qt::Key_Enter,     UKey_Return},  //used prior to next pair
  {Qt::Key_Return,    UKey_Return},

  {Qt::Key_Left,      UKey_Left},
  {Qt::Key_Up,        UKey_Up},
  {Qt::Key_Right,     UKey_Right},
  {Qt::Key_Down,      UKey_Down},
  {Qt::Key_Prior,     UKey_Prior},  // Key_PageUp = Key_Prior
  {Qt::Key_Next,      UKey_Next},   // Key_PageDown = Key_Next
  {Qt::Key_Home,      UKey_Home},
  {Qt::Key_End,       UKey_End},

  {Qt::Key_F1,        UKey_F1},
  {Qt::Key_F2,        UKey_F2},
  {Qt::Key_F3,        UKey_F3},
  {Qt::Key_F4,        UKey_F4},
  {Qt::Key_F5,        UKey_F5},
  {Qt::Key_F6,        UKey_F6},
  {Qt::Key_F7,        UKey_F7},
  {Qt::Key_F8,        UKey_F8},
  {Qt::Key_F9,        UKey_F9},   // Calendar
  {Qt::Key_F10,       UKey_F10},  // Address
  {Qt::Key_F11,       UKey_F11},  // Menu
  {Qt::Key_F12,       UKey_F12},  // Home
  {Qt::Key_F13,       UKey_F13},  // Mail
  {Qt::Key_F14,       UKey_F14},  // "Mail" being pressed long
  {Qt::Key_F15,       UKey_F15},
  {Qt::Key_F16,       UKey_F16},
  {Qt::Key_F17,       UKey_F17},
  {Qt::Key_F18,       UKey_F18},
  {Qt::Key_F19,       UKey_F19},
  {Qt::Key_F20,       UKey_F20},
  {Qt::Key_F21,       UKey_F21},  // Zenkaku/Hankaku
  {Qt::Key_F22,       UKey_F22},  // Fn
  {Qt::Key_F23,       UKey_F23},  // Fn + Space ("OnKun")
  {Qt::Key_F24,       UKey_F24},
  {Qt::Key_F25,       UKey_F25},  // Fn + Zenkaku/Hankaku ("Kanji")
  {Qt::Key_F26,       UKey_F26},  // Katakana/Hiragana toggle
  {Qt::Key_F27,       UKey_F27},
  {Qt::Key_F28,       UKey_F28},
  {Qt::Key_F29,       UKey_F29},
  {Qt::Key_F30,       UKey_F30},  // SELECT on SL-5500 (center of round cursor keys)
  {Qt::Key_F31,       UKey_F31},  // Fn + Katakana/Hiragana ("Kigou")
  {Qt::Key_F32,       UKey_F32},  // "Sync start" button on cradle
  {Qt::Key_F33,       UKey_F33},  // OK
  {Qt::Key_F34,       UKey_F34},  // Power (don't use this key)
  {Qt::Key_F35,       UKey_F35},  // "Calendar" being pressed long ("Sync" on SL-C700)

  // non-standard Zaurus specific keys
  {(Qt::Key)0x200f,   UKey_Private1},   // Fn + 1 ("Zoom out")
  {(Qt::Key)0x2010,   UKey_Private2},   // Fn + 2 ("Zoom in")
  {(Qt::Key)0x2012,   UKey_Private3},   // Fn + 3 ("Decrease contrast")
  {(Qt::Key)0x2011,   UKey_Private4},   // Fn + 4 ("Increase contrast")
  {(Qt::Key)0x201b,   UKey_Private5},   // Fn + 5
  {(Qt::Key)0x200a,   UKey_Private6},   // Fn + 6 ("Hiragana")
  {(Qt::Key)0x200b,   UKey_Private7},   // Fn + 7 ("Katakana")
  {(Qt::Key)0x200c,   UKey_Private8},   // Fn + 8 ("half-width Katakana")
  {(Qt::Key)0x2007,   UKey_Private9},   // Fn + 9 ("wide-width Alphanumeric")
  {(Qt::Key)0x2008,   UKey_Private10},  // Fn + 0 ("Alphanumeric")
  {(Qt::Key)0x2014,   UKey_Private11},  // Fn + Q
  {(Qt::Key)0x2015,   UKey_Private12},  // Fn + O
  {(Qt::Key)0x2016,   UKey_Private13},  // Fn + P
  {(Qt::Key)0x2017,   UKey_Private14},  // Fn + A
  {(Qt::Key)0x2018,   UKey_Private15},  // Fn + S
  {(Qt::Key)0x2022,   UKey_Private16},  // Fn + K
  {(Qt::Key)0x2019,   UKey_Private17},  // Fn + N
  {(Qt::Key)0x201a,   UKey_Private18},  // Fn + M

  // SL-6000 specific keys
  {(Qt::Key)0x2013,   UKey_Private19},  // "Backlight" being pressed long ("Rotate screen on the fly")
  {(Qt::Key)0x200d,   UKey_Private20},  // "Rec" being pressed long
  {(Qt::Key)0x200e,   UKey_Private21},  // user defined key (looks like "(  *  )")

#ifdef IMKIT_USE_QTE234_JAPANESE_IM_KEYS
  //{Qt::Key_Hiragana_Katakana, UKey_Hiragana_Katakana},  // Hiragana/Katakana toggle
  {Qt::Key_Hiragana_Katakana, UKey_Mode_switch},  // Hiragana/Katakana toggle?
  {Qt::Key_Henkan,            UKey_Henkan_Mode},  // Start/Stop Conversion
  {Qt::Key_Muhenkan,          UKey_Muhenkan},     // Cancel Conversion
  {Qt::Key_Zenkaku_Hankaku,   UKey_Zenkaku_Hankaku}  // Zenkaku/Hankaku toggle
#endif
};

static BidirMap<Qt::Key, UKeyModifier>::pair_t q2u_modifier_keycode_def[] = {
  {Qt::Key_Shift,   UMod_Shift},
  {Qt::Key_Control, UMod_Control},
  {Qt::Key_Alt,     UMod_Alt},
  {Qt::Key_Meta,    UMod_Meta}
};

static BidirMap<Qt::ButtonState, UKeyModifier>::pair_t
q2u_modifier_state_def[] = {
  {Qt::NoButton,      (UKeyModifier)0},
  {Qt::ShiftButton,   UMod_Shift},
  {Qt::ControlButton, UMod_Control},
  //{Qt::AltButton,     UMod_Meta},  //used prior to next pair
  {Qt::AltButton,     UMod_Alt}
};


BidirMap<Qt::Key, UKey>
UIMKeyEvent::q2u_keycode(q2u_keycode_def,
                           (sizeof(q2u_keycode_def)
                            / BidirMap<Qt::Key, UKey>::pair_size),
                         Qt::Key_unknown, (UKey)0);

BidirMap<Qt::Key, UKeyModifier>
UIMKeyEvent::q2u_modifier_keycode(q2u_modifier_keycode_def,
                                    (sizeof(q2u_modifier_keycode_def)
                                     / BidirMap<Qt::Key, UKeyModifier>::pair_size),
                                  Qt::Key_unknown, (UKeyModifier)0);

BidirMap<Qt::ButtonState, UKeyModifier>
UIMKeyEvent::q2u_modifier_state(q2u_modifier_state_def,
                                  (sizeof(q2u_modifier_state_def)
                                   / BidirMap<Qt::ButtonState, UKeyModifier>::pair_size),
                                Qt::NoButton, (UKeyModifier)0);


Qt::ButtonState
UIMKeyEvent::extract_modifier_state(QChar chr) {
  //TODO
  return Qt::NoButton;
}

UKeyModifier
UIMKeyEvent::q2u_convert_modifier_state(int qt_modifier_state) {
  int uim_modifier_state, a_modifier;
  BidirMap<Qt::ButtonState, UKeyModifier>::ordinary_map_t::const_iterator i;

  qt_modifier_state &= Qt::KeyButtonMask;

  uim_modifier_state = 0;
  for (i = q2u_modifier_state.ordinary_map.begin();
       i != q2u_modifier_state.ordinary_map.end();
       i++)
  {
    a_modifier = i->first;
    if (qt_modifier_state & a_modifier) {
      uim_modifier_state |= i->second;
    }
  }

  return (UKeyModifier)uim_modifier_state;
}

UIMKeyEvent::UIMKeyEvent(void)
  : accepted(true),
    keycode((UKey)0), modifiers((UKeyModifier)0), is_press(true)
{
}

UIMKeyEvent::UIMKeyEvent(UKey keycode_init, UKeyModifier modifiers_init,
                             bool is_press_init)
  : accepted(true),
    keycode(keycode_init), modifiers(modifiers_init), is_press(is_press_init)
{
}

UIMKeyEvent::UIMKeyEvent(const QKeyEvent &e)
  : accepted(true),
    keycode((UKey)0), modifiers((UKeyModifier)0), is_press(true)
{
  int qt_modifier_state;
  Qt::Key qt_keycode;
  QChar chr;

  qt_modifier_state = e.state();
  qt_keycode = (Qt::Key)e.key();
  chr = KeyEventConverter::extract_qchar(e);

  if (iscntrl(e.ascii()) && !isascii(qt_keycode)) {
    keycode = q2u_keycode.ordinary_map.lookup(qt_keycode);
  } else if (isascii(qt_keycode)
             && qt_keycode
             && (isgraph(chr.latin1())
                 || isspace(chr.latin1())
                 || iscntrl(chr.latin1())))
  {
    keycode = (UKey)imkit_strip_control(chr.latin1());
  } else {
    keycode = q2u_keycode.ordinary_map.lookup(qt_keycode);
  }
  modifiers = q2u_convert_modifier_state(qt_modifier_state);
  is_press = (e.type() == QEvent::KeyPress);
  accepted = e.isAccepted();
}

UIMKeyEvent::~UIMKeyEvent(void) {
}

//UIMKeyEvent::operator QKeyEvent(void) const {
//  //TODO
//}

bool
UIMKeyEvent::is_accepted(void) const {
  return accepted;
}

void
UIMKeyEvent::accept(void) {
  accepted = true;
}

void
UIMKeyEvent::ignore(void) {
  accepted = false;
}
