// IMKit: An inputmethod adaptation library for Qtopia environment
// Copyright (C) 2002-2004  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_1 $
// $Id: platform_qpe.cpp,v 1.29 2004/04/01 10:45:31 yamaken Exp $

// This file contains some code fragments from Qtopia GPL
// (qtopia-free-1.7.0/src/server/inputmethods.cpp). Original
// copyright is following.
/**********************************************************************
** Copyright (C) 2000-2002 Trolltech AS.  All rights reserved.
**
** This file is part of the Qtopia Environment.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include <qstringlist.h>
#include <qfont.h>
#include <qtranslator.h>
#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qregexp.h>
#include <qpe/qpeapplication.h>
#include <qpe/fontmanager.h>
#include <qpe/config.h>
#include "platform_qpe.h"


int IMKitGlobal::reentrance_count = 0;
int IMKitGlobal::plugin_enumeration_count = 0;
QMap<QString, int> IMKitGlobal::resources;
QStringList *IMKitGlobal::_enabled_plugins = NULL;
QString IMKitGlobal::orig_lang;
QString IMKitGlobal::fallback_lang;
QCopChannel *IMKitGlobal::channel = NULL;

static BidirMap<QString, InputMap>::pair_t imkit_str2sym_inputmap_def[] = {
  {"ALPHA",     IMKIT_INPUT_MAP_ALPHA},
  {"WALPHA",    IMKIT_INPUT_MAP_WALPHA},
  {"HIRAGANA",  IMKIT_INPUT_MAP_HIRAGANA},
  {"KATAKANA",  IMKIT_INPUT_MAP_KATAKANA},
  {"HKATAKANA", IMKIT_INPUT_MAP_HKATAKANA},
  {"KANJI",     IMKIT_INPUT_MAP_KANJI},
  {"RAW",       IMKIT_INPUT_MAP_RAW}
};

BidirMap<QString, InputMap>
imkit_str2sym_inputmap(imkit_str2sym_inputmap_def,
                       (sizeof(imkit_str2sym_inputmap_def)
                        / sizeof(imkit_str2sym_inputmap_def[0])),
                       "TERM", IMKIT_INPUT_MAP_TERM);


void
imkit_set_font(QWidget *widget) {
  QString lang;

  lang = getenv("LANG");
  if (lang != "ja" && FontManager::hasUnicodeFont()) {
    widget->setFont(FontManager::unicodeFont(FontManager::Proportional));
  }
}

void
imkit_set_font(QWidget *widget, const QString &lang) {
  QString group;
  QFont font;
  Config conf("IMKit");

  group = "Lang ";
  if (lang.isNull() || lang.isEmpty()) {
    group += "unknown";
  } else {
    group += lang;
  }
  conf.setGroup(group);

  if (conf.hasKey("Font")) {
    QStringList fontspec;
    QString family = "";
    int point_size = 12, weight = QFont::Normal, italic = 0;

    fontspec = conf.readListEntry("Font", ',');
    if (1 <= fontspec.count()) family = fontspec[0];
    if (2 <= fontspec.count()) point_size = fontspec[1].toInt();
    if (3 <= fontspec.count()) weight = fontspec[2].toInt();
    if (4 <= fontspec.count()) italic = fontspec[3].toInt();
    font = QFont(family, point_size, weight, italic);
  }
  widget->setFont(font);
}

int
IMKitGlobal::ref_resource(const QString &resource) {
  if (!resources.contains(resource)) {
    resources[resource] = 0;
  }
  return ++resources[resource];
}

int
IMKitGlobal::unref_resource(const QString &resource) {
  if (resources.contains(resource)) {
    if (!--resources[resource]) {
      resources.remove(resource);
      return 0;
    } else {
      return resources[resource];
    }
  } else {
    return 0;
  }
}

void
IMKitGlobal::ref(void) {
  reentrance_count++;
  plugin_enumeration_count++;

#ifndef IMKIT_QPE16_PLATFORM
  if (!channel) {
    channel = new QCopChannel("QPE/IME");
  }
#endif
}

void
IMKitGlobal::unref(void) {
  if (!--reentrance_count) {
    plugin_enumeration_count = 0;
    if (_enabled_plugins) {
      delete _enabled_plugins;
      _enabled_plugins = NULL;
    }
#ifndef IMKIT_QPE16_PLATFORM
    delete channel;
    channel = NULL;
#endif
  }  
}

#ifndef IMKIT_QPE16_PLATFORM
QUnknownInterface *
IMKitGlobal::create_instance(imkit_create_custom_ui_t ui_creator,
                             QPixmap *icon)
{
  InputMethodInterface *instance;

  //Q_CREATE_INSTANCE(IMKitImpl(ui_creator, icon))
  instance = new IMKitImpl(ui_creator, icon);

  return instance;
}
#endif  //IMKIT_QPE16_PLATFORM

QUnknownInterface *
IMKitGlobal::export_interface(imkit_create_custom_ui_t ui_creator,
                              QPixmap *icon)
{
  QUnknownInterface *instance;
  QUnknownInterface *iface = 0;

  IMKitGlobal::ref();
  instance = create_instance(ui_creator, icon);
  instance->queryInterface(IID_QUnknown, &iface);

  return iface;
}

#ifndef IMKIT_QPE16_PLATFORM
QStringList *
IMKitGlobal::enabled_plugins(void) {

  if (!_enabled_plugins) {
    QStringList plugin_files;
    QStringList::Iterator i;
    QString path, plugin_name;

    path = QPEApplication::qpeDir() + "/plugins/inputmethods";
    plugin_files = QDir(path, "libimkit-*.so").entryList();
    _enabled_plugins = new QStringList;
    for (i = plugin_files.begin(); i != plugin_files.end(); i++) {
      plugin_name = *i;
      plugin_name.replace(QRegExp("^lib"), "").replace(QRegExp("\\.so$"), "");
      _enabled_plugins->append(plugin_name);
    }
    qDebug("enabled_plugins(): methods = %s",
           (*_enabled_plugins).join(" ").latin1());
  }

  return _enabled_plugins;
}

QString
IMKitGlobal::currently_loading_plugin_name(void) {
  QString plugin;

  plugin = (*enabled_plugins())[plugin_enumeration_count - 1];
  qDebug("currently_loading_plugin_name = %s",  plugin.latin1());

  return plugin;
}
#endif  //IMKIT_QPE16_PLATFORM

QString
IMKitGlobal::translator_path(const QString &lang, const QString &im_name) {
  QString file = QString("lib") + im_name + ".qm";
  QString path = QPEApplication::qpeDir() + "i18n/" + lang + "/" + file;

  return path;
}

void
IMKitGlobal::setup_translator(const QString &im_name) {
#ifdef IMKIT_MISSING_TRANSLATION_SETUP
  QTranslator *trans;

  trans = new QTranslator(qApp);
  if (trans->load(translator_path(orig_lang, im_name))
      || trans->load(translator_path(fallback_lang, im_name)))
  {
    qApp->installTranslator(trans);
  } else {
    delete trans;
  }
#endif
}

void
IMKitGlobal::plugin_loading_has_been_started(const QString &plugin_lang) {
  {
    // setting fallback language while loading translator
    QString translator;

    orig_lang = getenv("LANG");
    // TODO: retrieve appropriate LANG for particular plugins
    fallback_lang = plugin_lang;

    translator = translator_path(orig_lang, currently_loading_plugin_name());
    if (!QFile(translator).exists()) {
      // setup fallback LANG for PluginLoader of Qtopia 1.6 or later
      qDebug("%s not found. fallback from %s into %s",
             translator.latin1(),
             orig_lang.latin1(),
             fallback_lang.latin1());
      setenv("LANG", fallback_lang, true);
    }
  }
}

void
IMKitGlobal::plugin_loading_has_been_finished(void) {
  {
    // resume original LANG after translator loaded
    setenv("LANG", IMKitGlobal::orig_lang, true);
    orig_lang = "";
    fallback_lang = "";
  }
}

void
IMKitGlobal::register_to_channel(QObject *receiver) {
  QObject::connect(channel, SIGNAL(received(const QCString&, const QByteArray&)),
                   receiver, SLOT(qcopReceive(const QCString&, const QByteArray&)));
}

KeyInjector::KeyInjector(QObject *parent, const char *name)
  : QObject(parent, name)
{
}

KeyInjector::~KeyInjector(void) {
}

bool
KeyInjector::filter(int unicode, int keycode, int modifiers,
                       bool is_press, bool autorepeat)
{
  QKeyEvent *e;
  bool stopped;

  e = keyevent_converter.create(unicode, keycode, modifiers,
                                is_press, autorepeat);
  e->ignore();
  output(e);
  stopped = e->isAccepted();
  delete e;
  
  return stopped;
}

void
KeyInjector::slot_activate(bool activity) {
  activate(activity);
}

void
QWSKeyBypassIn::on_activate(void) {
  if (!keyfilter_is_registered) {
    // status checking is required to prevent excessive filter
    // deletion on QWSServer::setKeyboardFilter(NULL)

    QWSServer::setKeyboardFilter(this);
    keyfilter_is_registered = true;
  }
}

void
QWSKeyBypassIn::on_deactivate(void) {
  if (keyfilter_is_registered) {
    // status checking is required to prevent excessive filter
    // deletion on QWSServer::setKeyboardFilter(NULL)

    QWSServer::setKeyboardFilter(NULL);
    keyfilter_is_registered = false;
  }
}

QWSKeyBypassIn::QWSKeyBypassIn(void)
  : KeyInjector(NULL, "QWSKeyBypassIn"), keyfilter_is_registered(false)
{
}

QWSKeyBypassIn::~QWSKeyBypassIn(void) {
}

void
QWSKeyBypassOut::on_input(QKeyEvent &e) {
  emit_as_signal(e);
  e.accept();
}

void
QWSKeyBypassOut::emit_as_signal(QKeyEvent &e) {
  keyevent_converter.convert(&e);
}

void
QWSKeyBypassOut::emit_key(int unicode, int scancode,
                          int modifiers,
                          bool is_press, bool autorepeat)
{
  emit key(unicode, scancode, modifiers, is_press, autorepeat);
}

QWSKeyBypassOut::QWSKeyBypassOut(void) : QObject(NULL, "QWSKeyBypassOut") {
  connect(&keyevent_converter, SIGNAL(converted(int, int, int, bool, bool)),
          this, SLOT(emit_key(int, int, int, bool, bool)));
}

QWSKeyBypassOut::~QWSKeyBypassOut(void) {
}

void
QWSKeyBypassOut::input(QKeyEvent *e) {
  if (is_active() && !e->isAccepted()) {
    on_input(*e);
  }
  output(e);
}

IMKitQCopReceiver::IMKitQCopReceiver(IMKitImpl *impl_init)
  : impl(impl_init)
{
}

void
IMKitQCopReceiver::qcopReceive(const QCString &msg, const QByteArray &data) {
  impl->qcopReceive(msg, data);
}

void
IMKitImpl::init(void) {
  if (!im_ui) {
    StdIMWidgetFactory *elem_factory;

    elem_factory = new StdIMWidgetFactory;
    im_ui = (*ui_creator)(elem_factory);
    prepare_key_bypass();
    join_to_bypass(im_ui->keyfilter_begin());
    QObject::connect(im_ui, SIGNAL(slot_activated(bool)),
                     bypass_in, SLOT(slot_activate(bool)));
    QObject::connect(im_ui, SIGNAL(slot_activated(bool)),
                     forwarder, SLOT(slot_activate(bool)));
    QObject::connect(im_ui, SIGNAL(slot_activated(bool)),
                     injector, SLOT(slot_activate(bool)));
  }
}

void
IMKitImpl::prepare_key_bypass(void) {
  bypass_in = new QWSKeyBypassIn;
  forwarder = new KeyHelperForwarder;
  injector = new KeyInjector;
  bypass_out = new QWSKeyBypassOut;
  *bypass_in << *forwarder << *injector << *bypass_out;
  bypass_out->activate();
}

void
IMKitImpl::join_to_bypass(CascadeKeyFilter *im_keyfilter) {
  if (!bypass_in) {
    prepare_key_bypass();
  }
  bypass_out->insert_before(im_keyfilter);
}

IMKitImpl::IMKitImpl(imkit_create_custom_ui_t ui_creator_init,
                     QPixmap *icon_init = 0)
  : ref(0), im_ui(NULL), ui_creator(ui_creator_init), _icon(icon_init),
    bypass_in(NULL), forwarder(NULL), injector(NULL), bypass_out(NULL),
    receiver(NULL)
{
  qDebug("IMKitImpl::IMKitImpl()");

  receiver = new IMKitQCopReceiver(this);
  IMKitGlobal::register_to_channel(receiver);
}

IMKitImpl::~IMKitImpl(void) {
  qDebug("IMKitImpl::~IMKitImpl()");
  delete receiver;
  delete bypass_out;
  delete injector;
  delete forwarder;
  delete bypass_in;
  delete _icon;
  if (im_ui) {
    delete im_ui;
  }
  IMKitGlobal::unref();
}

QWidget *
IMKitImpl::inputMethod(QWidget *parent, Qt::WFlags flags) {
  QWidget *imwidget;
  const char *language;

  IMKitGlobal::plugin_loading_has_been_finished();
  imwidget = im_ui->imwidget(parent, flags);
  language = im_ui->engine().language();
  imkit_set_font(imwidget, language);

  return imwidget;
}

void
IMKitImpl::resetState(void) {
#ifndef IMKIT_BROKEN_QPE_IMINTERFACE_RESET_FOR_COMPOSING_IM
  im_ui->reset_state();
#endif
}

QPixmap *
IMKitImpl::icon(void) {
  return _icon;
}

QString
IMKitImpl::name(void) {
  return im_ui->name();
}

void
IMKitImpl::onKeyPress(QObject *receiver, const char *slot) {
  QObject::connect(bypass_out, SIGNAL(key(ushort,ushort,ushort,bool,bool)),
                   receiver, slot);
}

// TODO: unify the code with ExtIMKitImpl::qcopReceive()
void
IMKitImpl::qcopReceive(const QCString &msg, const QByteArray &data) {
  QDataStream stream(data, IO_ReadOnly);

  if (!im_ui->imwidget()->isVisible()) {
    return;
  }

  if (msg == "imkitSetMode(QString)" || msg == "setMode(QString)") {
    QString map_name;
    InputMap map_val;

    stream >> map_name;
    map_val = imkit_str2sym_inputmap.ordinary_map.lookup(map_name);
    if (map_val == IMKIT_INPUT_MAP_TERM) return;

    im_ui->engine().select_map(map_val);
  } else if (msg == "imkitKeyEvent(int,int,int,int,int)"
             || msg == "keyEvent(int,int,int,int,int)")
  {
    int unicode, keycode, modifiers, is_press, autorepeat;

    stream >> unicode >> keycode >> modifiers >> is_press >> autorepeat;
    injector->filter(unicode, keycode, modifiers, is_press, autorepeat);
  } else if (msg == "imkitSetFont(QString,int,int,int)"
             || msg == "setFont(QString,int,int,int)")
  {
    QString family;
    int point_size, weight, italic;

    stream >> family >> point_size >> weight >> italic;
    QFont font(family, point_size, weight, italic);
    im_ui->imwidget()->setFont(font);
  }
}

QRESULT
IMKitImpl::queryInterface(const QUuid &uuid, QUnknownInterface **interface) {
  *interface = NULL;

  if (uuid == IID_QUnknown) {
    *interface = this;
    (*interface)->addRef();
    
    return QS_OK;
  } else if (uuid == IID_InputMethod) {
    init();
    if (im_ui->engine().name()) {
      QString im_name;

      *interface = this;
      (*interface)->addRef();
      IMKitGlobal::plugin_loading_has_been_started(im_ui->engine().language());
#ifdef IMKIT_MISSING_TRANSLATION_SETUP
      //Qtopia 1.5.0ˤtranslatoråȥåפΥɤޤޤƤ
      //widgetloadΤǵǽʤ
      im_name = IMKitGlobal::currently_loading_plugin_name();
      IMKitGlobal::setup_translator(im_name);
#endif
      
      return QS_OK;
    } else {
      return QS_FALSE;
    }
  } else {
    return QS_FALSE;
  }
}
