// IMKit: An inputmethod adaptation library for Qtopia environment
// Copyright (C) 2002-2004  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_1 $
// $Id: imwidget_std.cpp,v 1.16 2004/03/10 11:19:43 yamaken Exp $

#include <qobject.h>
#include <qobjectlist.h>
#include <qapplication.h>
#include "imwidget_std.h"


static bool candidate_win_should_be_dropdown(const QRect &focus);
static QPoint force_inside(const QRect &enclosure, const QRect &prisoner);


void
StdPreedit::connect_signals(void) {
  connect(this, SIGNAL(text_changed(const QString &)),
          widget(), SLOT(setText(const QString &)));
  connect(widget(), SIGNAL(selected(const QRect &)),
          this, SLOT(select(const QRect &)));
}

QString
StdPreedit::separator_str(void) {
  QString null_str = "";

  switch (state) {
  case IMKIT_PREEDIT_ST_CONV:
  case IMKIT_PREEDIT_ST_CSEG:
    return _separator_str;

  case IMKIT_PREEDIT_ST_NONE:
  case IMKIT_PREEDIT_ST_EDIT:
  default:
    return null_str;
  }
}

StdPreedit::StdPreedit(const QString &separator_str_init = "|")
  : adaptee(NULL), inputmask_filter(NULL), state(IMKIT_PREEDIT_ST_NONE),
    _cursor_position(0), _separator_str(separator_str_init)
{
}

StdPreedit::StdPreedit(PreeditWidget *adaptee_init,
                       const QString &separator_str_init = "|")
  : adaptee(NULL), inputmask_filter(NULL), state(IMKIT_PREEDIT_ST_NONE),
    _cursor_position(0), _separator_str(separator_str_init)
{
  set_widget(adaptee_init);
}

StdPreedit::~StdPreedit(void) {
  if (widget()) {
    widget()->removeEventFilter(inputmask_filter);
  }
  if (inputmask_filter) {
    delete inputmask_filter;
  }
  if (adaptee) delete adaptee;
}

QWidget *
StdPreedit::widget(void) {
  return ((adaptee) ? adaptee->widget() : NULL);
}

void
StdPreedit::set_widget(PreeditWidget *new_adaptee) {
  if (!inputmask_filter) {
    inputmask_filter = new InputMaskFilter;
  }

  if (widget()) {
    widget()->removeEventFilter(inputmask_filter);
  }
  adaptee = new_adaptee;
  widget()->installEventFilter(inputmask_filter);
  connect_signals();
}

int
StdPreedit::cursor_position(void) {
  return _cursor_position;
}

void
StdPreedit::set_cursor_position(int new_position) {
  _cursor_position = new_position;
  adaptee->setCursorPosition(_cursor_position);
}

void
StdPreedit::set_focus(int focus_begin, int focus_length) {
  if (focus_length) {
    adaptee->setSelection(focus_begin, focus_length);
  }
}

void
StdPreedit::set_text(const QString &new_text) {
  bool text_is_updated;

  text_is_updated = (new_text != text);
  if (text_is_updated) {
    text = new_text;
    emit text_changed(text);
  }
}

void
StdPreedit::set_separator_str(const QString &new_separator_str = "|") {
  _separator_str = new_separator_str;
}

void
StdPreedit::update_state(PreeditState new_state) {
  state = new_state;
}

void
StdPreedit::update(Segments &segments) {
  Segments::iterator segment;
  Segment::Type prev_segment_type;
  QString new_text, separator;
  int prev_cursor_pos, cursor_pos, focus_begin, focus_end, focus_length;

  separator = separator_str();
  prev_segment_type = Segment::IMKIT_SEG_NORMAL;
  prev_cursor_pos = cursor_position();
  cursor_pos = 0;
  focus_begin = focus_end = 0;
  focus_length = 0;
  new_text = "";

  for (segment = segments.begin(); segment != segments.end(); segment++) {
    int pos;
    uint segment_length;

    pos = new_text.length();
    segment_length = (*segment)->str().length();
    switch ((*segment)->type()) {
    case Segment::IMKIT_SEG_CURSOR:
      cursor_pos = pos;
      break;

    case Segment::IMKIT_SEG_SEPARATOR:
      new_text += separator;
      break;

    case Segment::IMKIT_SEG_FOCUS:
      // focus over multiple segments is supported
      if (prev_segment_type != Segment::IMKIT_SEG_FOCUS) {
        focus_begin = pos;
      }
      focus_length += segment_length;
      focus_end = focus_begin + focus_length;
      cursor_pos = focus_begin;
      new_text += (*segment)->str();
      break;

    case Segment::IMKIT_SEG_NORMAL:
    case Segment::IMKIT_SEG_EDITING:
    case Segment::IMKIT_SEG_PENDING:
    default:
      new_text += (*segment)->str();
      break;
    }
    prev_segment_type = (*segment)->type();
  }

  set_text(new_text);

  //եϰϤwidgetϤ߽ФΤɤ
  set_cursor_position(cursor_pos + focus_length);

  set_cursor_position(cursor_pos);
  set_focus(focus_begin, focus_length);
}

void
StdPreedit::select(const QRect &selection) {
  QRect abs_focus;
  QPoint abs_pos;

  abs_pos = widget()->mapToGlobal(selection.topLeft());
  abs_focus = QRect(abs_pos.x(), abs_pos.y(),
                    selection.width(), selection.height());

  emit focus_changed(abs_focus);
}

void
StdInputMapIndicator::connect_signals(void) {
  connect(widget(), SIGNAL(activated(int)),
          this, SLOT(select_by_adaptee(int)));
}

StdInputMapIndicator::StdInputMapIndicator(void) 
  : adaptee(NULL), _state(IMKIT_INPUT_MAP_ALPHA), n_state(0), pixmaps(NULL)
{
}

StdInputMapIndicator::
StdInputMapIndicator(InputMapIndicatorWidget *adaptee_init,
                     PixmapRepository *pixmaps_init)
  : adaptee(NULL), _state(IMKIT_INPUT_MAP_ALPHA), n_state(0),
    pixmaps(pixmaps_init)
{
  set_widget(adaptee_init);
}

StdInputMapIndicator::~StdInputMapIndicator(void) {
  if (adaptee) delete adaptee;
}

QWidget *
StdInputMapIndicator::widget(void) {
  return ((adaptee) ? adaptee->widget() : NULL);
}

void
StdInputMapIndicator::set_widget(InputMapIndicatorWidget *new_adaptee) {
  adaptee = new_adaptee;
  connect_signals();
}

InputMap
StdInputMapIndicator::state(void) const {
  return _state;
}

void
StdInputMapIndicator::add_state(InputMap map_code, const QString &label) {
  QPixmap icon;
  int index;

  index = n_state++;
  if (pixmaps) {
    icon = pixmaps->lookup(map_code);
  }
  if (!icon.isNull()) {
    adaptee->insertItem(icon, label, index);
  } else {
    adaptee->insertItem(label, index);
  }
  n2a_map_code.add_map(index, map_code);
}

void
StdInputMapIndicator::select_by_adaptee(int index) {
  _state = n2a_map_code.ordinary_map.lookup(index);
  emit selected(_state);
}

void
StdInputMapIndicator::select(InputMap new_state) {
  int index;

  _state = new_state;
  index = n2a_map_code.inverse_map.lookup(_state);
  adaptee->setCurrentItem(index);
}

void
StdInputMapIndicator::clear(void) {
  adaptee->clear();
  n2a_map_code.init_map(NULL, 0);
  n_state = 0;
  _state = IMKIT_INPUT_MAP_ALPHA;
}

void
StdCandidateWindow::connect_signals(void) {
  if (widget()) {
    connect(widget(), SIGNAL(clicked(QListBoxItem *)),
            this, SLOT(select_by_widget(QListBoxItem *)));
  }
}

void
StdCandidateWindow::select_by_widget(QListBoxItem *selected_item) {
  int _index;

  _index = adaptee->index(selected_item);
  select_by_widget(_index);
}

void
StdCandidateWindow::select_by_widget(int new_index) {
  index = new_index;
  emit selected(index);
}

StdCandidateWindow::StdCandidateWindow(void)
  : adaptee(NULL), index(0), n_candidates(0)
{
}

StdCandidateWindow::StdCandidateWindow(CandidateWindowWidget *adaptee_init)
  : adaptee(NULL), index(0), n_candidates(0)
{
  set_widget(adaptee_init);
}

StdCandidateWindow::~StdCandidateWindow(void) {
  if (adaptee) delete adaptee;
}

QWidget *
StdCandidateWindow::widget(void) {
  //return adaptee;  //template <class Adaptee = QListBox>ǽˤʤä
  return ((adaptee) ? adaptee->widget() : NULL);
}

void
StdCandidateWindow::set_widget(CandidateWindowWidget *new_adaptee) {
  adaptee = new_adaptee;
  connect_signals();
}

void
StdCandidateWindow::slot_activate(bool activity) {
  activate(activity);
}

void
StdCandidateWindow::update(Candidates &new_candidates) {
  Candidates::iterator i;

  adaptee->clear();
  for (i = new_candidates.begin(); i != new_candidates.end(); i++) {
    adaptee->insertItem((*i)->str());
  }
  n_candidates = new_candidates.size();

#ifndef IMKIT_FIXED_SIZE_CANDIDATE_WINDOW
  //widget()->adjustSize();
#endif
}

void
StdCandidateWindow::hilight(int new_index, bool reversely) {
  index = new_index;
  adaptee->setCurrentItem(index);

  //if (reversely) {
  //  adaptee->setBottomItem(index);
  //} else {
  //  adaptee->setTopItem(index);
  //}
}

static bool
candidate_win_should_be_dropdown(const QRect &focus) {
  int screen_height, upper_margin, lower_margin;

  screen_height = imkit_screen_size().height();
  upper_margin = focus.top();
  lower_margin = screen_height - focus.bottom() - 1;

  return (upper_margin <= lower_margin);
}

//TODO: QtƱδؿʤĴ٤
static QPoint
force_inside(const QRect &enclosure, const QRect &prisoner) {
  int new_x, new_y;

  new_x = min(enclosure.right(), prisoner.right()) - prisoner.width() + 1;
  new_x = max(enclosure.left(), new_x);
  new_y = min(enclosure.bottom(), prisoner.bottom()) - prisoner.height() + 1;
  new_y = max(enclosure.top(), new_y);

  return QPoint(new_x, new_y);
}

void
StdCandidateWindow::move_adjacent(const QRect &focus) {
  QPoint compensated_position;
  QRect screen(QPoint(0, 0), imkit_screen_size());
  QRect candidate_win_rect(focus.bottomLeft(), widget()->frameSize());
  int y;

  if (candidate_win_should_be_dropdown(focus)) {
    y = focus.bottom() + 1;  //window posision is outside of the focus
  } else {
    y = focus.top() - candidate_win_rect.height();
  }
  candidate_win_rect.moveTopLeft(QPoint(focus.x(), y));
  compensated_position = force_inside(screen, candidate_win_rect);
  widget()->move(compensated_position);
}

void
StdIMWidget::setFont(const QFont &new_font) {
  QWidget::setFont(new_font);
  // top-level widget ignores ParentFontChange
  if (children()) {
    QObjectListIt i(*children());
    QObject *child;

    for(; child = i.current(); ++i) {
      if (child->isWidgetType()) {
        ((QWidget *)child)->setFont(new_font);
      }
    }
  }
}

#ifndef IMKIT_USE_TEMPLATE_QOBJECT
IMKIT_DEFINE_ACTIVATED_SIGNALS_FOR_QWIDGET(StdIMWidget);
#endif  //IMKIT_USE_TEMPLATE_QOBJECT
