// IMKit: An inputmethod adaptation library for Qtopia environment
// Copyright (C) 2002-2004  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_3 $
// $Id: imwidget_std.h,v 1.11 2004/03/16 15:46:37 yamaken Exp $

#ifndef IMKIT_IMWIDGET_STD_H
#define IMKIT_IMWIDGET_STD_H

#include <qguardedptr.h>
#include <qcombobox.h>
#include <qlistbox.h>
#include <qhbox.h>
#include "imwidget.h"


template <class Widget>
class PreeditWidgetAbstractor : public PreeditWidget {
protected:
  QGuardedPtr<Widget> concrete_widget;

public:
  PreeditWidgetAbstractor(void);
  PreeditWidgetAbstractor(Widget *concrete_widget_init);
  virtual QWidget *widget(void);
  virtual void set_widget(Widget *new_widget);
  virtual void setCursorPosition(int new_position);
  virtual void setSelection(int focus_begin, int focus_length);

//public slots:
//  void setText(const QString &);

//signals:
//  void selected(const QRect &selection);
};


class StdPreedit : public Preedit {
  Q_OBJECT

protected:
  PreeditWidget *adaptee;
  QGuardedPtr<QObject> inputmask_filter;
  PreeditState state;
  int _cursor_position;
  QString _separator_str;
  QString text;

protected:
  virtual void connect_signals(void);
  virtual QString separator_str(void);

signals:
  void cursor_position_changed(int new_position);
  void focus_changed(int focus_begin, int focus_length);
  void text_changed(const QString &new_text);
  void focus_changed(const QRect &abs_focus);

public:
  StdPreedit(const QString &separator_str_init = "|");
  StdPreedit(PreeditWidget *adaptee_init,
             const QString &separator_str_init = "|");
  virtual ~StdPreedit(void);
  virtual QWidget *widget(void);
  virtual void set_widget(PreeditWidget *new_adaptee);
  virtual int cursor_position(void);
  virtual void set_cursor_position(int new_position);
  virtual void set_focus(int focus_begin, int focus_length);
  virtual void set_text(const QString &new_text);
  virtual void set_separator_str(const QString &new_separator_str = "|");

public slots:
  virtual void update_state(PreeditState new_state);
  virtual void update(Segments &segments);
  virtual void select(const QRect &selection);
};


template <class Widget>
class InputMapIndicatorWidgetAbstractor : public InputMapIndicatorWidget {
protected:
  QGuardedPtr<Widget> concrete_widget;

public:
  InputMapIndicatorWidgetAbstractor(void);
  InputMapIndicatorWidgetAbstractor(Widget *concrete_widget_init);
  virtual QWidget *widget(void);
  virtual void set_widget(Widget *new_widget);
  virtual void setCurrentItem(int index);
  virtual void insertItem(const QPixmap &pixmap, const QString &label,
                          int index);
  virtual void insertItem(const QString &label, int index);
  virtual void clear(void);
};


class StdInputMapIndicator : public InputMapIndicator {
  Q_OBJECT

protected:
  InputMapIndicatorWidget *adaptee;
  InputMap _state;
  int n_state;
  PixmapRepository *pixmaps;
  BidirMap<int, InputMap> n2a_map_code;

protected:
  virtual void connect_signals(void);

public:
  StdInputMapIndicator(void);
  StdInputMapIndicator(InputMapIndicatorWidget *adaptee_init,
                       PixmapRepository *pixmaps_init = NULL);
  virtual ~StdInputMapIndicator(void);
  virtual QWidget *widget(void);
  virtual void set_widget(InputMapIndicatorWidget *new_adaptee);
  virtual InputMap state(void) const;

signals:
  void selected(InputMap new_state);

protected slots:
  virtual void select_by_adaptee(int index);

public slots:
  virtual void add_state(InputMap map_code, const QString &label);
  virtual void select(InputMap new_state);
  virtual void clear(void);
};


class CandidateWindowWidget
//: virtual public QWidget  //Qt2ǤϥݡȤƤʤ
{
public:
  virtual ~CandidateWindowWidget(void) {}
  virtual QWidget *widget(void) = 0;  // : virtual public QWidget
  virtual void clear(void) = 0;
  virtual void insertItem(const QString &item) = 0;
  virtual void setCurrentItem(int index) = 0;
  virtual void setTopItem(int index) = 0;
  virtual void setBottomItem(int index) = 0;

  //TODO: QListBoxItem˰¸ʤ褦ˤ
  virtual int index(const QListBoxItem *item) = 0;
  virtual void resize(void) = 0;
};


template <class Widget>
class CandidateWindowWidgetAbstractor : public CandidateWindowWidget {
protected:
  QGuardedPtr<Widget> concrete_widget;
  int n_candidates;
  int cell_spacing;       // TODO: be obsoleted
  int decoration_height;  // TODO: be obsoleted
  int width;              // TODO: be obsoleted

public:
  CandidateWindowWidgetAbstractor(void);
  CandidateWindowWidgetAbstractor(Widget *widget_init);
  virtual QWidget *widget(void);
  virtual void set_widget(Widget *new_widget);
  virtual void clear(void);
  virtual void insertItem(const QString &item);
  virtual void setCurrentItem(int index);
  virtual void setTopItem(int index);
  virtual void setBottomItem(int index);
  virtual int index(const QListBoxItem *item);
  virtual void resize(void);
  virtual void configure(int new_n_candidates, int new_cell_spacing,
                         int new_decoration_height, int new_width);
};


class StdCandidateWindow
  : public CandidateWindow, public ActivatableQWidgetAdapter
{
  Q_OBJECT

protected:
  CandidateWindowWidget *adaptee;
  Candidates::size_type index, n_candidates;

protected:
  virtual void connect_signals(void);

protected slots:
  virtual void select_by_widget(QListBoxItem *selected_item);
  virtual void select_by_widget(int new_index);

public:
  StdCandidateWindow(void);
  StdCandidateWindow(CandidateWindowWidget *adaptee_init);
  virtual ~StdCandidateWindow(void);
  virtual QWidget *widget(void);
  virtual void set_widget(CandidateWindowWidget *new_adaptee);

public slots:
  virtual void slot_activate(bool activity);
  virtual void update(Candidates &new_candidates);
  virtual void hilight(int selected, bool reversely);
  virtual void move_adjacent(const QRect &focus);
};


#ifdef IMKIT_USE_TEMPLATE_QOBJECT

typedef QWidgetWithActivationSignal<QHBox> StdIMWidget;

#else  //IMKIT_USE_TEMPLATE_QOBJECT

class StdIMWidget : public QHBox {
  Q_OBJECT

public:
  inline StdIMWidget(QWidget *parent = 0,
                     const char *name = 0, WFlags flags = 0)
    : QHBox(parent, name, flags)
  {}
  virtual void setFont(const QFont &new_font);

#ifndef IMKIT_COMPILE_WITH_PREPROCESSABLE_MOC
protected:
  virtual void showEvent(QShowEvent *e);
  virtual void hideEvent(QHideEvent *e);
signals:
  void activated(bool activity);
#else
  IMKIT_DECLARE_ACTIVATED_SIGNALS_FOR_QWIDGET;
#endif
};

#endif  //IMKIT_USE_TEMPLATE_QOBJECT


//////////////////////////////////////////////////////////////////////////////

template <class Widget>
PreeditWidgetAbstractor<Widget>::PreeditWidgetAbstractor(void)
  : concrete_widget(NULL)
{
}

template <class Widget>
PreeditWidgetAbstractor<Widget>::
PreeditWidgetAbstractor(Widget *concrete_widget_init)
  : concrete_widget(NULL)
{
  set_widget(concrete_widget_init);
}

template <class Widget>
QWidget *
PreeditWidgetAbstractor<Widget>::widget(void) {
  return concrete_widget;
}

template <class Widget>
void
PreeditWidgetAbstractor<Widget>::set_widget(Widget *new_widget) {
  concrete_widget = new_widget;
}

template <class Widget>
void
PreeditWidgetAbstractor<Widget>::setCursorPosition(int new_position) {
  //(concrete_widget->hasFocus() == true)¸ʤưʤǤ
  //QLineEditǤ̵
  concrete_widget->setCursorPosition(new_position);
}

template <class Widget>
void
PreeditWidgetAbstractor<Widget>::
setSelection(int focus_begin, int focus_length) {
  //եɽȴǼ¸뤿selectionήѤ
  concrete_widget->setSelection(focus_begin, focus_length);
}

template <class Widget>
InputMapIndicatorWidgetAbstractor<Widget>::
InputMapIndicatorWidgetAbstractor(void)
  : concrete_widget(NULL)
{
}

template <class Widget>
InputMapIndicatorWidgetAbstractor<Widget>::
InputMapIndicatorWidgetAbstractor(Widget *concrete_widget_init)
  : concrete_widget(NULL)
{
  set_widget(concrete_widget_init);
}

template <class Widget>
QWidget *
InputMapIndicatorWidgetAbstractor<Widget>::widget(void) {
  return concrete_widget;
}

template <class Widget>
void
InputMapIndicatorWidgetAbstractor<Widget>::set_widget(Widget *new_widget) {
  concrete_widget = new_widget;
}

template <class Widget>
void
InputMapIndicatorWidgetAbstractor<Widget>::setCurrentItem(int index) {
  concrete_widget->setCurrentItem(index);
}

template <class Widget>
void
InputMapIndicatorWidgetAbstractor<Widget>::
insertItem(const QPixmap &pixmap, const QString &label, int index) {
  concrete_widget->insertItem(pixmap, label, index);
}

template <class Widget>
void
InputMapIndicatorWidgetAbstractor<Widget>::
insertItem(const QString &label, int index) {
  concrete_widget->insertItem(label, index);
}

template <class Widget>
void
InputMapIndicatorWidgetAbstractor<Widget>::clear(void) {
  concrete_widget->clear();
}


template <class Widget>
CandidateWindowWidgetAbstractor<Widget>::CandidateWindowWidgetAbstractor(void)
  : concrete_widget(NULL),
    n_candidates(10), cell_spacing(2), decoration_height(2), width(160)
{
}

template <class Widget>
CandidateWindowWidgetAbstractor<Widget>::
CandidateWindowWidgetAbstractor(Widget *widget_init)
  : n_candidates(10), cell_spacing(2), decoration_height(2), width(160)
{
  set_widget(widget_init);
}

template <class Widget>
QWidget *
CandidateWindowWidgetAbstractor<Widget>::widget(void) {
  return concrete_widget;
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::set_widget(Widget *new_widget) {
  concrete_widget = new_widget;
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::clear(void) {
  concrete_widget->clear();
  //concrete_widget->setCurrentItem(-1);  // no effect
  resize();
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::insertItem(const QString &item) {
  concrete_widget->insertItem(item);
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::setCurrentItem(int index) {
  bool moving_forward, acrossing_boundary;
  int current_index, tail_index;

  current_index = concrete_widget->currentItem();
  tail_index = concrete_widget->count() - 1;
  acrossing_boundary = ((current_index == 0 && index == tail_index)
                        || (current_index == tail_index && index == 0));
  moving_forward = (!acrossing_boundary && current_index <= index
                    || acrossing_boundary && index == 0);
  
  concrete_widget->setSelected(index, true);
  if (moving_forward && (index % n_candidates == 0)) {
    setTopItem(index);
  } else if (!moving_forward && (index % n_candidates == n_candidates - 1)) {
    setBottomItem(index);
  } else if (!moving_forward && acrossing_boundary) {
    setTopItem((tail_index / n_candidates) * n_candidates);
  }
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::setTopItem(int index) {
  concrete_widget->setTopItem(index);
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::setBottomItem(int index) {
  concrete_widget->setBottomItem(index);
}

template <class Widget>
int
CandidateWindowWidgetAbstractor<Widget>::index(const QListBoxItem *item) {
  return concrete_widget->index(item);
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::resize(void) {
#ifdef IMKIT_FIXED_SIZE_CANDIDATE_WINDOW
  QFontMetrics fm(concrete_widget->font());
  int height;

  height = (fm.height() + cell_spacing) * n_candidates + decoration_height;
  concrete_widget->setResizePolicy(QListBox::Manual);
  concrete_widget->resize(width, height);
#else
#if 0
  //concrete_widget->setResizePolicy(QListBox::AutoOneFit);
#else
  concrete_widget->setResizePolicy(QListBox::Manual);
  concrete_widget->resize(concrete_widget->sizeHint());
#endif
#endif  // IMKIT_FIXED_SIZE_CANDIDATE_WINDOW
}

template <class Widget>
void
CandidateWindowWidgetAbstractor<Widget>::configure(int new_n_candidates,
                                                   int new_cell_spacing,
                                                   int new_decoration_height,
                                                   int new_width)
{
  if (0 < new_n_candidates && new_n_candidates < 100) {
    n_candidates = new_n_candidates;
  }
  if (0 <= new_cell_spacing && new_cell_spacing < 100) {
    cell_spacing = new_cell_spacing;
  }
  if (0 <= new_decoration_height && new_decoration_height < 100) {
    decoration_height = new_decoration_height;
  }
  if (0 < new_width && new_width <= 4096) {
    width = new_width;
  }
}

#endif  //IMKIT_IMWIDGET_STD_H
