// IMKit-uim: A Qtopia InputMethod interface for uim
// Copyright (C) 2002-2004  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_4 $
// $Id: engine_uim.cpp,v 1.32 2004/04/14 12:04:27 yamaken Exp $

#include <ctype.h>
#include <string.h>
#include <qobject.h>
#include <qcstring.h>
#include "uim_keyevent.h"
#include "engine_uim.h"
#include "platform_qpe.h"

#define IMKIT_UIM_DEFAULT_ENCODING "UTF-8"
//#define IMKIT_UIM_USE_UIM_CONVERTER  // requires uim API spec change

#ifdef UIM_INPUT_MAP_INVALID
#error "UIM_INPUT_MAP_INVALID is predefined"
#else
#define UIM_INPUT_MAP_INVALID -0x10000
#endif

uim_context UIMEngine::global_uim_ctx = NULL;

#if 0
static BidirMap<int, PreeditState>::pair_t n2a_preedit_state_def[] = {
  {UIM_PUSHBACK_ATTR_CONV, IMKIT_PREEDIT_ST_CONV},
  {UIM_PUSHBACK_ATTR_CSEG, IMKIT_PREEDIT_ST_CSEG},
  {UIM_PUSHBACK_ATTR_EDIT, IMKIT_PREEDIT_ST_EDIT},
  {UIM_PUSHBACK_ATTR_NONE, IMKIT_PREEDIT_ST_NONE}
};

BidirMap<int, PreeditState>
UIMEngine::n2a_preedit_state(n2a_preedit_state_def,
                               (sizeof(n2a_preedit_state_def)
                                / BidirMap<int, PreeditState>::pair_size),
                               UIM_PUSHBACK_ATTR_NONE, IMKIT_PREEDIT_ST_NONE);
#endif


#ifdef IMKIT_UIM_USE_UIM_CONVERTER
static struct uim_code_converter imkit_uim_conv = {
  &UIMCodeConverter::is_convertible,
  &UIMCodeConverter::create,
  &UIMCodeConverter::convert,
  &UIMCodeConverter::release
};

UIMCodeConverter::UIMCodeConverter(const QString &convert_to_init,
                                   const QString &convert_from)
  : convert_to(convert_to_init), codec(NULL), fallback_converter(NULL)
{
  if (is_natively_convertible(convert_to, convert_from)) {
    codec = QTextCodec::codecForName(convert_from);
  } else if (uim_iconv->is_convertible("UTF-8", convert_from)) {
    fallback_converter = uim_iconv->create("UTF-8", convert_from);
  }
}

UIMCodeConverter::~UIMCodeConverter(void) {
  if (fallback_converter) {
    uim_iconv->release(fallback_converter);
  }
}

char *
UIMCodeConverter::convert(const char *str) {
  char *utf8str = NULL;

  if (fallback_converter) {
    utf8str = uim_iconv->convert(fallback_converter, str);
  }  

  if (convert_to == "QString") {
    QString *qstr = NULL;

    if (codec) {
      qstr = new QString(codec->toUnicode(str));
    } else if (fallback_converter) {
      qstr = new QString(QString::fromUtf8(utf8str));
    }

    return (char *)qstr;
  } else if (convert_to == "UTF-8") {
    if (codec) {
      utf8str = qstrdup(codec->toUnicode(str).utf8());
    }

    return utf8str;
  } else {
    return NULL;
  }
}

int
UIMCodeConverter::is_natively_convertible(const char *tocode,
                                          const char *fromcode)
{
  QString to(tocode);
  bool codec_is_installed, result;

  codec_is_installed = (QTextCodec::codecForName(fromcode) != NULL);
  result = (to == "QString" || to == "UTF-8") ? codec_is_installed : false;

  return result;
}

int
UIMCodeConverter::is_convertible(const char *tocode, const char *fromcode) {
  return (is_natively_convertible(tocode, fromcode)
          || uim_iconv->is_convertible(tocode, fromcode));
}

void *
UIMCodeConverter::create(const char *tocode, const char *fromcode) {
  if (is_convertible(tocode, fromcode)) {
    return new UIMCodeConverter(tocode, fromcode);
  } else {
    return NULL;
  }
}

char *
UIMCodeConverter::convert(void *obj, const char *str) {
  UIMCodeConverter *converter;

  converter = (UIMCodeConverter *)obj;

  return converter->convert(str);
}

void
UIMCodeConverter::release(void *obj) {
  UIMCodeConverter *converter;

  converter = (UIMCodeConverter *)obj;
  delete converter;
}

struct uim_code_converter *
UIMCodeConverter::create_uim_if(void) {
  struct uim_code_converter *uim_if;

  uim_if = new struct uim_code_converter;
  uim_if->is_convertible = is_convertible;
  uim_if->create = create;
  uim_if->convert = convert;
  uim_if->release = release;

  return uim_if;
}
#endif  // IMKIT_UIM_USE_UIM_CONVERTER


UIMSegment::UIMSegment(int attr_init, const QString &qstr_init = "")
  : attr(attr_init), qstr(qstr_init)
{
}

UIMSegment::~UIMSegment(void) {
}

const QString &
UIMSegment::str(void) {
  return qstr;
}

UIMSegment::Type
UIMSegment::type(void) const {
  if (attr & UPreeditAttr_Reverse) {
    return IMKIT_SEG_FOCUS;
  } else if (attr & UPreeditAttr_Cursor) {
    return IMKIT_SEG_CURSOR;
  } else if (attr & UPreeditAttr_Separator) {
    return IMKIT_SEG_SEPARATOR;
  } else if (attr & UPreeditAttr_UnderLine) {
    return IMKIT_SEG_EDITING;
  } else {
    return IMKIT_SEG_NORMAL;
  }
}

UIMCandidate::UIMCandidate(const QString &qstr_init = "")
  : qstr(qstr_init)
{
}

UIMCandidate::~UIMCandidate(void) {
}

const QString &
UIMCandidate::str(void) {
  return qstr;
}

const QTextCodec *
UIMEngine::codec(void) const {
  QTextCodec *_codec;

  _codec = QTextCodec::codecForName(encoding);
  if (!_codec) {
    _codec = QTextCodec::codecForName(IMKIT_UIM_DEFAULT_ENCODING);
  }

  return _codec;
}

UIMEngine::UIMEngine(void)
  : uim_ctx(NULL), segments(NULL), candidates(NULL),
    n2a_inputmap(NULL, 0, UIM_INPUT_MAP_INVALID, IMKIT_INPUT_MAP_TERM),
  _command_map(NULL), encoding(NULL), _lang(NULL), _name(NULL)
{
  int err;

  err = uim_init();
  if (!global_uim_ctx) {
    global_uim_ctx = uim_create_context(this, IMKIT_UIM_DEFAULT_ENCODING,
                                        NULL, NULL,
#ifdef IMKIT_UIM_USE_UIM_CONVERTER
                                        &imkit_uim_conv,
#else
                                        uim_iconv,
#endif
                                        cb_commit);
  }

  uim_engine_name = indicated_uim_engine_name();
  if (available_uim_ims().contains(uim_engine_name)) {
    QString qstr_name;

    qstr_name = QString("uim-") + uim_engine_name;
    _name = strdup(qstr_name.latin1());
    if (_name == "uim-anthy") {
      IMKitGlobal::ref_resource("anthy");
    }
    
    init_state();
    sync();
  }
  IMKitGlobal::ref_resource("uim");
}

UIMEngine::~UIMEngine(void) {
  free_state();
  imkit_delete_command_map(_command_map);
  if (_name == "uim-anthy") {
    IMKitGlobal::unref_resource("anthy");
    // IMKit-Anthy must be released precede uim-anthy
  }
  free(_name);
  if (!IMKitGlobal::unref_resource("uim")) {
    uim_release_context(global_uim_ctx);
    global_uim_ctx = NULL;
    uim_quit();
  }
}

void
UIMEngine::reset(void) {
  free_state();
  init_state();
  //uim_reset_context(uim_ctx);
  init_segments();
  init_candidates();
  sync();
}

void
UIMEngine::init_state(void) {
  select_im(NULL, uim_engine_name.latin1());
}

void
UIMEngine::free_state(void) {
  if (uim_ctx) {
    uim_release_context(uim_ctx);
    uim_ctx = NULL;
  }
  imkit_delete_ref_container<Segments>(&segments);
  imkit_delete_ref_container<Candidates>(&candidates);
}

void
UIMEngine::init_segments(void) {
  imkit_init_ref_container<Segments>(&segments);
}

void
UIMEngine::init_candidates(void) {
  imkit_init_ref_container<Candidates>(&candidates);
}

void
UIMEngine::update_inputmap(void) {
  int i, n_states;

  n2a_inputmap.init_map(NULL, 0);
  emit map_states_cleared();
  n_states = uim_get_nr_modes(uim_ctx);
  for (i = 0; i < n_states; i++) {
    InputMap abstract_state;
    QString label;

#ifdef IMKIT_UIM_USE_UIM_CONVERTER
    label = *(QString *)uim_get_mode_name(uim_ctx, i);
#else
    label = codec()->toUnicode(uim_get_mode_name(uim_ctx, i));
#endif

    abstract_state = (InputMap)((int)IMKIT_INPUT_MAP_RUNTIME_VAL_BASE + i);
    if (label == tr("RAW") || label == tr("chokusetsu-nyuuryoku")) {
      abstract_state = IMKIT_INPUT_MAP_ALPHA;
    } else if (label == tr("hiragana") || label == tr("TUT-hi")) {
      abstract_state = IMKIT_INPUT_MAP_HIRAGANA;
    } else if (label == tr("katakana") || label == tr("TUT-ka")) {
      abstract_state = IMKIT_INPUT_MAP_KATAKANA;
    } else if (label == tr("hankaku-katakana")) {
      abstract_state = IMKIT_INPUT_MAP_HKATAKANA;
    } else if (label == tr("zenkaku-eisuu")) {
      abstract_state = IMKIT_INPUT_MAP_WALPHA;
    } else if (label == tr("tcode")
               || label == tr("py")
               || label == tr("pyunihan"))
    {
      abstract_state = IMKIT_INPUT_MAP_KANJI;
    }

    n2a_inputmap.add_map(i, abstract_state);
    emit map_state_added(abstract_state, label);
  }
  emit map_changed(map_state());
}

const char *
UIMEngine::name(void) const {
  return _name;
}

const char *
UIMEngine::language(void) const {
  return _lang;
}

InputMap
UIMEngine::map_state(void) const {
  int native_map_state;

  native_map_state = uim_get_current_mode(uim_ctx);

  return n2a_inputmap.ordinary_map.lookup(native_map_state);
}

CommandMap *
UIMEngine::command_map(void) {
  typedef TemplateCommandWithKeyEvent<UIMEngine> command_t;
  command_t *cmd;

  if (!_command_map) {
    cmd = new command_t(this,
                        &UIMEngine::cmd_receive_keyevent,
                        "receive_keyevent");

    _command_map = new CommandMap;
    (*_command_map)["receive_keyevent"] = cmd;
  }

  return _command_map;
}

void
UIMEngine::cmd_receive_keyevent(QKeyEvent &e) {
  UIMKeyEvent uim_e(e);
  bool ignored = true;

  if (uim_e.keycode) {
    if (uim_e.is_press) {
      ignored = uim_press_key(uim_ctx, uim_e.keycode, uim_e.modifiers);
    } else {
      ignored = uim_release_key(uim_ctx, uim_e.keycode, uim_e.modifiers);
    }
  }
  if (!ignored) {
    e.accept();
  }
}

void
UIMEngine::input_str(const QString &str) {
  //TODO
}

void
UIMEngine::input_char(QChar chr) {
  //TODO
}

int
UIMEngine::get_im_index(const char *engine) {
  const char *nth_engine;
  int i;

  for (i = 0; i < uim_get_nr_im(global_uim_ctx); i++) {
    nth_engine = uim_get_im_name(global_uim_ctx, i);
    if (!strcmp(nth_engine, engine)) {
      return i;
    }
  }

  return -1;
}

void
UIMEngine::select_im(InputMap nth) {
  const char *lang, *engine;

  if ((int)nth < uim_get_nr_im(global_uim_ctx)) {
    lang = uim_get_im_language(global_uim_ctx, (int)nth);
    engine = uim_get_im_name(global_uim_ctx, (int)nth);
  } else {
    lang = engine = NULL;
  }
  select_im(lang, engine);
}

void
UIMEngine::select_im(const char *lang, const char *engine) {
  int nth_im;

  if (uim_ctx) {
    uim_release_context(uim_ctx);
  }
  nth_im = get_im_index(engine);
#ifdef IMKIT_UIM_USE_UIM_CONVERTER
  encoding = "QString";
#else
  encoding = uim_get_im_encoding(global_uim_ctx, nth_im);
  if (!QTextCodec::codecForName(encoding, 1)) {
    qDebug("select_im: encoding '%s' not found in QTextCodecs."
           " fallback to %s to use iconv",
           encoding, IMKIT_UIM_DEFAULT_ENCODING);
    encoding = IMKIT_UIM_DEFAULT_ENCODING;
  }
#endif
  if (!lang) {
    lang = uim_get_im_language(global_uim_ctx, nth_im);
  }
  if (!strlen(lang)) {
    lang = NULL;
  }
  _lang = lang;
  qDebug("select_im: encoding = '%s', lang = '%s', engine = '%s'",
         encoding, lang, engine);

  uim_ctx = uim_create_context(this,
                               (char *)encoding, (char *)lang, (char *)engine,
#ifdef IMKIT_UIM_USE_UIM_CONVERTER
                               &imkit_uim_conv,
#else
                               uim_iconv,
#endif
                               cb_commit);
  uim_set_preedit_cb(uim_ctx, cb_clear, cb_pushback, cb_update);
  uim_set_mode_list_update_cb(uim_ctx, cb_set_mode_list_update);
  uim_set_mode_cb(uim_ctx, cb_set_mode);
  uim_set_candidate_selector_cb(uim_ctx,
                                cb_candidate_win_activate,
                                cb_candidate_win_update_focus,
                                cb_candidate_win_shift_page,
                                cb_candidate_win_deactivate);
  init_segments();
  init_candidates();

  update_inputmap();
}

void
UIMEngine::select_map(InputMap new_map) {
  int native;

  native = n2a_inputmap.inverse_map.lookup(new_map);
  if (native == UIM_INPUT_MAP_INVALID) return;

  uim_set_mode(uim_ctx, native);
  emit map_changed(new_map);
}

void
UIMEngine::select_candidate(Candidates::size_type nth) {
  uim_set_candidate_index(uim_ctx, nth);
}

void
UIMEngine::cb_commit(void *_this, const char *str) {
  ((UIMEngine *)_this)->slot_commit(str);
}

void
UIMEngine::cb_clear(void *_this) {
  ((UIMEngine *)_this)->slot_clear();
}

void
UIMEngine::cb_pushback(void *_this, int attr, const char *str) {
  ((UIMEngine *)_this)->slot_pushback(attr, str);
}

void
UIMEngine::cb_update(void *_this) {
  ((UIMEngine *)_this)->slot_update();
}

void
UIMEngine::cb_set_mode(void *_this, int mode) {
  ((UIMEngine *)_this)->slot_set_mode(mode);
}

void
UIMEngine::cb_set_mode_list_update(void *_this) {
  ((UIMEngine *)_this)->slot_set_mode_list_update();
}

void
UIMEngine::cb_candidate_win_activate(void *_this, int nr, int display_limit) {
  ((UIMEngine *)_this)->slot_candidate_win_activate(nr, display_limit);
}

void
UIMEngine::cb_candidate_win_update_focus(void *_this, int index) {
  ((UIMEngine *)_this)->slot_candidate_win_update_focus(index);
}

void
UIMEngine::cb_candidate_win_shift_page(void *_this, int direction) {
  ((UIMEngine *)_this)->slot_candidate_win_shift_page(direction);
}

void
UIMEngine::cb_candidate_win_deactivate(void *_this) {
  ((UIMEngine *)_this)->slot_candidate_win_deactivate();
}

void
UIMEngine::slot_commit(const char *str) {
  QString unicode_str;

#ifdef IMKIT_UIM_USE_UIM_CONVERTER
  unicode_str = *(QString *)str;
#else
  unicode_str = codec()->toUnicode(str);
#endif
  emit committed(unicode_str);
}

void
UIMEngine::slot_clear(void) {
  init_segments();
}

void
UIMEngine::slot_pushback(int attr, const char *str) {
  QString qstr;

#ifdef IMKIT_UIM_USE_UIM_CONVERTER
  qstr = *(QString *)str;
#else
  qstr = codec()->toUnicode(str);
#endif
  segments->push_back(new UIMSegment(attr, qstr));
}

void
UIMEngine::slot_update(void) {
  sync();
}

void
UIMEngine::slot_set_mode(int mode) {
  InputMap abstract_mode;

  abstract_mode = n2a_inputmap.ordinary_map.lookup(mode);
  emit map_changed(abstract_mode);
}

void
UIMEngine::slot_set_mode_list_update(void) {
  update_inputmap();
}

void
UIMEngine::slot_candidate_win_activate(int nr, int display_limit) {
  int i, accel_enumeration_hint;
  uim_candidate candidate;
  QString candidate_str;

  init_candidates();
  accel_enumeration_hint = 0; // TODO: enumerate properly
  for (i = 0; i < nr; i++) {
    candidate = uim_get_candidate(uim_ctx, i, accel_enumeration_hint);
#ifdef IMKIT_UIM_USE_UIM_CONVERTER
    candidate_str = *(QString *)uim_candidate_get_cand_str(candidate);
#else
    candidate_str = codec()->toUnicode(uim_candidate_get_cand_str(candidate));
#endif
    candidates->push_back(new UIMCandidate(candidate_str));
    uim_candidate_free(candidate);
  }

  emit update_candidates(*candidates);
  //slot_candidate_win_update_focus(initial_index);
  emit activation_hint_for_candidates(true);
}

void
UIMEngine::slot_candidate_win_update_focus(int index) {
  emit hilight_candidate(index, false);
}

void
UIMEngine::slot_candidate_win_shift_page(int direction) {
  //TODO: implement
}

void
UIMEngine::slot_candidate_win_deactivate(void) {
  emit activation_hint_for_candidates(false);
}

QStringList
UIMEngine::available_uim_ims(void) {
  int i;
  const char *engine;
  QStringList result;

  for (i = 0; i < uim_get_nr_im(global_uim_ctx); i++) {
    engine = uim_get_im_name(global_uim_ctx, i);
    result += QString(engine);
  }
  result.remove("default");  // ignore useless 'default' IM

  return result;
}

QString
UIMEngine::indicated_uim_engine_name(void) {
  QString plugin_name;

  plugin_name = IMKitGlobal::currently_loading_plugin_name();
  return plugin_name.replace(QRegExp("^imkit-uim-"), "");
}

PreeditState
UIMEngine::preedit_state(void) const {
  Segments::const_iterator segment;

  for (segment = segments->begin(); segment != segments->end(); segment++) {
    if ((*segment)->type() == Segment::IMKIT_SEG_FOCUS) {
      return IMKIT_PREEDIT_ST_CONV;
    }
  }
  return IMKIT_PREEDIT_ST_EDIT;
}

void
UIMEngine::sync(void) {
  emit preedit_state_changed(preedit_state());
  emit segments_changed(*segments);
}
