// IMKit: An inputmethod adaptation library for Qtopia environment
// Copyright (C) 2002-2004  YamaKen <yamaken@bp.iij4u.or.jp>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// $Name: IMKIT_0_4_4 $
// $Id: keyfilter.cpp,v 1.13 2004/04/29 14:41:01 yamaken Exp $

#include <ctype.h>
#include "keyfilter.h"


void
imkit_delete_command_map(CommandMap *command_map) {
  CommandMap::iterator pair;

  if (command_map) {
    for (pair = command_map->begin(); pair != command_map->end(); pair++) {
      delete pair->second;
    }
    delete command_map;
  }
}

unsigned char
imkit_strip_control(unsigned char chr) {
  if (iscntrl(chr)) {
    if (chr == 0) {
      chr = ' ';
    } else if (chr != 0x7f) {
      chr += '`';
    }
  }

  return chr;
}

unsigned char
imkit_strip_modifier(unsigned char chr) {
  if (isupper(chr)) {
    chr = tolower(chr);
  } else {
    chr = imkit_strip_control(chr);
  }

  return chr;
}

QChar
KeyEventConverter::extract_qchar(const QKeyEvent &e) {
  QChar chr;

  chr = (e.text().length()) ? e.text()[0] : QChar('\0');

  return chr;
}

KeyEventConverter::KeyEventConverter(void)
  : QObject(NULL, "KeyEventConverter")
{
}

KeyEventConverter::~KeyEventConverter(void) {
}

QKeyEvent *
KeyEventConverter::create(int unicode, int keycode, int modifiers,
                          bool is_press, bool autorepeat)
{
  QKeyEvent *e;

  e = new QKeyEvent(((is_press) ? QEvent::KeyPress : QEvent::KeyRelease),
                    keycode,
                    QChar(unicode).latin1(),
                    modifiers,
                    QString(QChar(unicode)),
                    autorepeat);

  return e;
}

void
KeyEventConverter::create_by_signal(int unicode, int keycode, int modifiers,
                                    bool is_press, bool autorepeat)
{
  QKeyEvent *e;

  e = create(unicode, keycode, modifiers, is_press, autorepeat);
  e->ignore();

  emit created(e);
}

void
KeyEventConverter::convert(QKeyEvent *e) {
  int unicode, modifiers;
  bool is_press;

  unicode = e->text().at(0).unicode();
  modifiers = e->state();
  is_press = (e->type() == QEvent::KeyPress);

  emit converted(unicode, e->key(), modifiers, is_press, e->isAutoRepeat());
}

void
CascadeKeyFilter::on_input(QKeyEvent &e) {
}

void
CascadeKeyFilter::output(QKeyEvent *e) {
  if (_next) {
    _next->input(e);
  }
}

CascadeKeyFilter::CascadeKeyFilter(void)
  : _prev(NULL), _next(NULL)
{
}

CascadeKeyFilter::CascadeKeyFilter(const CascadeKeyFilter &orig)
  : _prev(orig._prev), _next(orig._next)
{
}

CascadeKeyFilter::~CascadeKeyFilter(void) {
  remove();
}

void
CascadeKeyFilter::insert(CascadeKeyFilter *to_be_prev,
                         CascadeKeyFilter *to_be_next,
                         CascadeKeyFilter *other)
{
  CascadeKeyFilter *end_of_other;

  for (end_of_other = other;
       end_of_other->_next;
       end_of_other = end_of_other->_next);

  if (to_be_prev) {
    to_be_prev->_next = other;
    other->_prev = to_be_prev;
  }
  if (to_be_next) {
    end_of_other->_next = to_be_next;
    to_be_next->_prev = end_of_other;
  }
}

void
CascadeKeyFilter::remove(CascadeKeyFilter *begin, CascadeKeyFilter *end) {
  if (!begin || !end) return;

  if (begin->_prev) {
    begin->_prev->_next = end->_next;
  }
  if (end->_next) {
    end->_next->_prev = begin->_prev;
  }

  begin->_prev = NULL;
  end->_next = NULL;
}

CascadeKeyFilter *
CascadeKeyFilter::prev(void) {
  return _prev;
}

CascadeKeyFilter *
CascadeKeyFilter::next(void) {
  return _next;
}

void
CascadeKeyFilter::insert_before(CascadeKeyFilter *other) {
  insert(_prev, this, other);
}

void
CascadeKeyFilter::insert_after(CascadeKeyFilter *other) {
  insert(this, _next, other);
}

CascadeKeyFilter &
CascadeKeyFilter::operator<<(CascadeKeyFilter &other) {
  CascadeKeyFilter *i, *end_of_other;

  insert_after(&other);
  for (i = &other; i->next(); i = i->next());
  end_of_other = i;

  return *end_of_other;
}

void
CascadeKeyFilter::remove(void) {
  remove(this, this);
}

void
CascadeKeyFilter::remove_till(CascadeKeyFilter *end) {
  remove(this, end);
}

void
CascadeKeyFilter::input(QKeyEvent *e) {
  bool is_dummy_input;

  is_dummy_input = (e->key() == IMKIT_NON_UNICODE_CHAR
                    && e->ascii() == 0
                    && e->state() == 0);
  if (is_active() && !e->isAccepted() && !is_dummy_input) {
    on_input(*e);

    if (e->isAccepted()) {
      QEvent::Type type;
      QKeyEvent *dummy_input_to_prevent_screen_off;

      type = e->type();
      dummy_input_to_prevent_screen_off
        = new QKeyEvent(type, IMKIT_NON_UNICODE_CHAR, 0, 0);
    
      e = dummy_input_to_prevent_screen_off;
      e->ignore();
      output(e);
      delete e;

      return;
    }
  }

  output(e);
}

CharInjector::CharInjector(void) : QObject(NULL, "CharInjector") {
}

CharInjector::~CharInjector(void) {
}

void
CharInjector::inject_char(QChar chr) {
  QString text;
  uchar ascii;
  QKeyEvent *keypress, *keyrelease;

  text = chr;
  ascii = chr.latin1();

  keypress = new QKeyEvent(QEvent::KeyPress, 0, ascii, 0, text, false);
  keypress->ignore();
  output(keypress);
  delete keypress;

  keyrelease = new QKeyEvent(QEvent::KeyRelease, 0, ascii, 0, text, false);
  keyrelease->ignore();
  output(keyrelease);
  delete keyrelease;
}

void
CharInjector::inject_str(const QString &str) {
  uint i;
  QChar chr;

  for (i = 0; i < str.length(); i++) {
    chr = str[i];
    inject_char(chr);
  }
}

void
CharInjector::commit(const QString &str) {
  inject_str(str);
}

Command::~Command(void) {
}

DoNothingCommand::~DoNothingCommand(void) {
}

const char *
DoNothingCommand::name(void) const {
  return _name;
}

void
DoNothingCommand::execute(QKeyEvent &e) {
}
