package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * すかいらーくの住所を取得するクラスです。
 * @author Kumano Tatsuo
 * @since 3.16
 * 2005/12/04
 */
public class Skylark {
	/**
	 * 最初のURL
	 */
	private static final String URL1 = "http://www2.info-mapping.com/skylark/map/list2.asp?chihou=";

	/**
	 * 次のURL
	 */
	private static final String URL2 = "&ctyname=";

	/**
	 * ベースURL
	 */
	private static final String BASE_URL = "http://www2.info-mapping.com/skylark/map/";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "SJIS";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator + "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "skylark_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * すかいらーくの座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param cityID 市区町村コード
	 * @param cityName 市区町村名
	 * @param prefectureID 都道府県コード
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String cityID, final String cityName,
		final String prefectureID, final Map<String, Point2D> isj, final MapPanel panel) throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Skylark.CACHE_DIR).exists()) {
			new File(Skylark.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Skylark.CACHE_DIR + File.separator + Skylark.PREFIX + cityID
			+ Skylark.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(cacheFile), "SJIS");
			ISJUtil.parseAddresses(Skylark.getAddresses(cityName, prefectureID, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(new FileInputStream(new File(
			cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				for (final String attribute : new String[] { "すかいらーく", "バーミヤン", "ガスト", "夢庵" }) {
					if (items[1].startsWith(attribute)) {
						ret.put(new Point2D.Double(x, y), attribute);
						break;
					}
				}
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * すかいらーくの市区町村別ページを解析して住所の一覧を取得します。
	 * @param cityName 市区町村名
	 * @param prefectureID 都道府県コード
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @since 3.16
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(final String cityName,
		final String prefectureID, final MapPanel panel) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL1 + prefectureID + URL2
				+ URLEncoder.encode(cityName, Skylark.ENCODING));
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
			final Pattern pattern = Pattern
				.compile("<a href=\"separate.asp\\?chihou=[0-9]+&ctyname=[^<>]+&ID=([0-9]+)\">[^<>]+</a>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					final URL url2 = new URL(BASE_URL + "map.asp?ID=" + matcher.group(1));
					panel.addMessage(url2 + "をダウンロードしています。");
					ret.putAll(getAddresses(url2, panel));
					panel.removeMessage();
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
		}
		return ret;
	}

	/**
	 * すかいらーくの店舗別ページを解析して住所と店舗名の対応を取得します。
	 * @param url URL
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応
	 * @throws IOException 
	 */
	public static Map<String, String> getAddresses(final URL url, final MapPanel panel) throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		final Scanner scanner = new Scanner(new InputStreamReader(url.openStream(), ENCODING));
		final Pattern pattern1 = Pattern.compile("<font class=\"shopname\"><b>([^<>]+)</b></font>");
		final Pattern pattern2 = Pattern.compile("住所：([^<>]+)");
		String caption = null;
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final Matcher matcher1 = pattern1.matcher(line);
			if (matcher1.find()) {
				caption = matcher1.group(1);
			}
			final Matcher matcher2 = pattern2.matcher(line);
			if (matcher2.find()) {
				if (caption != null) {
					ret.put(matcher2.group(1), caption);
					break;
				}
			}
		}
		scanner.close();
		return ret;
	}
}
