package shop;

import isj.ISJUtil;
import java.awt.geom.Point2D;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import map.MapPanel;

/**
 * 吉野家の住所を取得するクラスです。
 * @author Kumano Tatsuo
 * 2005/12/03
 */
public class Yoshinoya {
	/**
	 * 最初のURL
	 * この後に都道府県コードが来ます。
	 */
	private static final String URL1 = "http://vip.mapion.co.jp/c/f?vp=10&p=1&grp=yoshinoya&uc=21&ob=0&mx=20&bool=admi2code&pg=&sfn=yoshinoya_search-partinfo-list_00&oi=admi3code&admi2code=";

	/**
	 * 次のURL
	 * この後に市区町村コードが来ます。
	 */
	private static final String URL2 = "&admi3=";

	/**
	 * エンコーディング
	 */
	private static final String ENCODING = "EUC-JP";

	/**
	 * キャッシュを保存するディレクトリ
	 */
	private static final String CACHE_DIR = ".map" + File.separator
		+ "shops";

	/**
	 * キャッシュファイル名の接頭語
	 */
	private static final String PREFIX = "yoshinoya_";

	/**
	 * キャッシュファイル名の接尾語
	 */
	private static final String SUFFIX = ".csv";

	/**
	 * 吉野家の座標と店舗名の対応表をホームページ又はキャッシュから取得します。
	 * @param cityID 市区町村コード
	 * @param cityName 市区町村名
	 * @param isj 街区レベル位置参照情報
	 * @param panel 地図を描画するパネル
	 * @return 座標と店舗名の対応表
	 * @throws IOException 
	 */
	public static Map<Point2D, String> load(final String cityID,
		final String cityName, final Map<String, Point2D> isj, final MapPanel panel)
		throws IOException {
		final Map<Point2D, String> ret = new LinkedHashMap<Point2D, String>();
		if (!new File(Yoshinoya.CACHE_DIR).exists()) {
			new File(Yoshinoya.CACHE_DIR).mkdirs();
		}
		final String cacheFile = Yoshinoya.CACHE_DIR + File.separator
			+ Yoshinoya.PREFIX + cityID + Yoshinoya.SUFFIX;
		if (!new File(cacheFile).exists()) {
			final PrintWriter out = new PrintWriter(new File(
				cacheFile), "SJIS");
			ISJUtil.parseAddresses(Yoshinoya.getAddresses(cityID,
				cityName, panel), out, isj);
			out.close();
		}
		final Scanner scanner = new Scanner(new InputStreamReader(
			new FileInputStream(new File(cacheFile)), "SJIS"));
		while (scanner.hasNextLine()) {
			final String line = scanner.nextLine();
			final String[] items = line.split(",");
			if (items.length == 4) {
				final double x = Double.parseDouble(items[2]);
				final double y = Double.parseDouble(items[3]);
				ret.put(new Point2D.Double(x, y), "吉牛");
			} else {
				System.out.println("WARNING: データ形式が不正です。" + line);
			}
		}
		scanner.close();
		return ret;
	}

	/**
	 * 吉野家の市区町村別ページを解析して住所の一覧を取得します。
	 * @param cityID 市区町村コード
	 * @param cityName 市区町村名
	 * @param panel 地図を描画するパネル
	 * @return 住所と店舗名の対応表
	 * @throws IOException 
	 * @throws UnsupportedEncodingException 
	 */
	public static Map<String, String> getAddresses(
		final String cityID, final String cityName, final MapPanel panel)
		throws IOException {
		final Map<String, String> ret = new LinkedHashMap<String, String>();
		try {
			final URL url = new URL(URL1 + cityID.substring(0, 2)
				+ URL2 + cityID);
			panel.addMessage(url + "をダウンロードしています。");
			final Scanner scanner = new Scanner(
				new InputStreamReader(url.openStream(), ENCODING));
			String caption = null;
			final Pattern pattern = Pattern
				.compile("<a href=[^<>]+>([^<>]+)</a>");
			final Pattern pattern2 = Pattern
				.compile("<td [^<>]+>([^<>]+)</td>");
			while (scanner.hasNextLine()) {
				final String line = scanner.nextLine();
				final Matcher matcher = pattern.matcher(line);
				if (matcher.find()) {
					caption = matcher.group(1);
				}
				final Matcher matcher2 = pattern2.matcher(line);
				if (matcher2.find()) {
					if (caption != null) {
						final String address = matcher2.group(1);
						if (address.contains(cityName)) {
							ret.put(address, caption);
						}
					}
				}
			}
			scanner.close();
			panel.removeMessage();
		} catch (FileNotFoundException e) {
		}
		return ret;
	}
}
